/******************************************************************************
 * (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
 ******************************************************************************/

/* ---------------------------------------------------------------
 * File: xdynamic.c
 *       Dynamic Capabilities setup, delete and query routines
 * -----------------------------------------------------------------*/

#include <xtypedef.h>

#include <xdynamic.h>
#include <xerrcapi.h>
#include <xiocommo.h>
#include <xladata.h>
#include <xregx10.h>
#include <xsession.h>
#include <xutil.h>

#include <xstreq.h>

#include <xpattdef.h>

#include <xstrutil.h>

#ifdef CUSTOM_OEM1
#include <oem.h>
#endif

/* avoid warnings when using names.h */
#ifdef __BORLANDC__
#  pragma warn -use
#endif

/* names.h contains the stringlists that are generated from the
   *defs.h files in ../include by parsnames.cl */
#define BX_COMPILING_NAMESFILE
#include <xnames.h>
#undef BX_COMPILING_NAMESFILE

#define DO_STANDARD_ERROR_HANDLING           1
#define DO_NO_ERROR_HANDLING                 0

/* make adding of parameters a little more readable... */
#define __ADDPAR(name,shortname,a3,a4,a5,a6,a7,prop,a9,a10,a11) \
 BX_ERRCHECK (BestXAddParam (&p_anchor, num_elem++,name,shortname,#prop,a3,a4,a5,a6,a7,(bx_int32) prop,a9,a10,a11))

#define ADDPAR(a) __ADDPAR a

/* ------------------------------------------------------------------------
 * Data fields
 * Note: Leave the NULL at the end of the Stringlists. The compiler needs
 * it.
 * ------------------------------------------------------------------------ */

/* ------------------------------------------------------------------------
 *  Xact Compiler String Lists follow here
 * ------------------------------------------------------------------------ */
/* extern const bx_param_stringlisttype byten_table[]; */

/* extern const bx_param_stringlisttype yesno_table[]; */

/* ------------------------------------------------------------------------
 * ------------------------------------------------------------------------
 * Some things are solved using Tables for the different Hardwares
 * These things are signals for trace memory and pattern terms.
 * ------------------------------------------------------------------------ */

/* the layout of the signaltable is as follows: (squeeze it into bx_paraminfo)
 * new:  name, nameshort (always NULL), startbitpos (in min_val),
 * width (in max_val), default, NULL, shiftvalue (in paramrule).
 * the remaining infos within the struct are not entered here, the compiler
 * just leaves them uninitialized, but we do not need them
 * the official table in the analyzer class is built there */
/* every list needs a terminating line (important !! ) */

extern bx_param_infotype e2929a_signal_table[];
extern bx_param_infotype e2930a_signal_table[];

extern bx_patt_infotype e2929a_patt_signal_table[];
extern bx_patt_infotype e2930a_patt_signal_table[];



/* the memory map tables */
extern bx_memmaptype Mephisto_RI_Blk2Hw[];
extern bx_memmaptype Mephisto_RI_Beh2Hw[];
extern bx_memmaptype Mephisto_RT_Beh2Hw[];
extern bx_memmaptype Mephisto_CI_Beh2Hw[];
extern bx_memmaptype Mephisto_CT_Beh2Hw[];






/* --------------------------------------------------------------------------
 * We use malloc so many times in here that this is worth it.
 * -------------------------------------------------------------------------- */
static bx_errtype malloc_chk_zero(void **ptr, size_t size)
{
  if (NULL==(*ptr = BestXMemMalloc(size)))
  {
    return (BX_E_HOST_MEM_FULL);
  }

  BESTX_MEMSET(*ptr, 0, size);
  return (BX_E_OK);
}


/* ------------------------------------------------------------------------
 * This fct reads the capabilities out of the hardware
 * ------------------------------------------------------------------------ */
bx_errtype EXPORT BestXIHWCapabilityGet(bx_handletype handle, bx_int32 * lic)
{
  bx_errtype err;
  bx_int8 buf[12];
  bx_int16 buflen = OUT_LICENSE_GET;
  bx_hwtype theBoard;

#ifdef CUSTOM_OEM1  
  int i ;
  deviceCache_t  *localDeviceCache = NULL ;
  
  for( i = 0 ; i < actualHandles ; i++ )
  {
    // If the entries match
    if( deviceCache[i].portnum == bx_handlearray[handle].entered_port )
    {
        // Remember the entry and bail
        localDeviceCache = &deviceCache[i] ;
        break ;
    }
  }
  // We now either have a match or free entry
#endif

  /* determine the license here.
   * for the old e2925 and e2925 with deep memory the license is fixed and
   * set to normal CAPI Access (see below).
   * The Ichiban boards get the license from the hardware directly
   */

  err = BX_E_OK;

  BestXGetHWFromHandle(handle, &theBoard);

  if (bx_handlearray[handle].port == BX_PORT_OFFLINE)
  {
    *lic = (bx_int32)((bx_int64) bx_handlearray[handle].portnumber & ~BX_CAPABILITY_HARDWARE_MASK);
  }
  else
  {

#ifdef CUSTOM_OEM1
    if( ( localDeviceCache ) && ( localDeviceCache->lic ) )
    {
        *lic = localDeviceCache->lic ;
    }
    else
#endif
    {
      switch(theBoard)
      {
        case BX_HW_E2929A:
        case BX_HW_E2929A_DEEP:
        case BX_HW_E2929B:
        case BX_HW_E2929B_DEEP:
        case BX_HW_E2930A:
        case BX_HW_E2930B:
          if (bx_handlearray[handle].OpenType!=BX_OPEN_FAST)
          {
            err = BestXBasicCommand(handle, CMD_LICENSE_GET, NULL, IN_LICENSE_GET,
                  buf, &buflen);
            (void) BestXStream2Long(lic, buf, 3UL);
            break;
          }
        /* no break here intentionally ! */
        case BX_HW_E2922A:
        case BX_HW_E2922B:
        case BX_HW_E2923A:
        case BX_HW_E2923B:
        case BX_HW_E2923F:
          *lic=(BX_CAPABILITY_EXERCISER | BX_CAPABILITY_OBSERVER | BX_CAPABILITY_CAPI);
          break;

        case BX_HW_UNKNOWN:
          err=BX_E_UNKNOWN_HARDWARE;
          break;

        default: 
          err=BX_E_INVALID_CASE;
      } /*switch*/
    }

#ifdef CUSTOM_OEM1
    if( ( localDeviceCache ) && ( ! localDeviceCache->lic ) )
    {
       localDeviceCache->lic = *lic ;
    }
#endif

  } /*else */

  BX_ERRETURN(err);
}
/* ------------------------------------------------------------------------
 * This fct writes the license back to the hardware
 * ------------------------------------------------------------------------ */
bx_errtype EXPORT BestXIHWCapabilitySet(bx_handletype handle, bx_int32 * lic)
{
  bx_errtype err;
  bx_int8 buf[12];

  if (bx_handlearray[handle].port == BX_PORT_OFFLINE)
  {
    err = BX_E_WRONG_HW;
  }
  else
  {
    (void) BestXLong2Stream(buf, lic, 3UL);
    if(BestXHasFirmware(handle))
    {
      err = BestXBasicCommand(handle, CMD_LICENSE_SET, buf, IN_LICENSE_SET,
      NULL, NULL);
    }
  }

  BX_ERRETURN(err);
}


/* ------------------------------------------------------------------------
 * This function adds  a new parameter to the list of already available
 * parameters. In case of the first access, it starts with a completely new
 * list of parameters.
 * ------------------------------------------------------------------------ */
bx_errtype EXPORT BestXAddParam(
    bx_param_infotype ** anchor, /* add the data to this list */
    bx_int32 num_elems,          /* how many elements do we have without the
                                 * new one */
    bx_charptrtype param_name,   /* namestring of standard form */
    bx_charptrtype param_short,  /* namestring of the short form */
    bx_ccharptrtype param_propname,    /* namestring: CAPI form (BX_...)  */
    bx_int32 min_val,            /* maximum value also bitstart(signal) */
    bx_int32 max_val,            /* minimum value also bitend (signal) */
    bx_int32 defaultval,         /* default value of this parameter */
    bx_charptrtype defaultstr,   /* default string of this parameter */
    bx_int32 paramrule,          /* param might have a special rule */
    bx_int32 proptyp,            /* property for the C-API call (union) */
    const bx_param_stringlisttype * stringlist,  /* array of possible
                                                 * stringlists */
    bx_int32 grouptyp,         /* what group the property belongs to */
    const bx_memmaptype * memmap /* memory map (for block/behavior) */
)
{
  bx_param_infotype *pLastAnchor;
  if (*anchor == NULL)           /* uninitialized field? */
  {
    num_elems = 0;
  }

  *anchor = (bx_param_infotype *) BestXMemRealloc((void *) (*anchor),
    (size_t) ((num_elems + 1) * sizeof(bx_param_infotype)));
  if (*anchor == NULL)
  {
    return (BX_E_HOST_MEM_FULL); /* could not allocate memory */
  }

  pLastAnchor = &((*anchor)[(size_t) num_elems]);

  pLastAnchor->param_name = param_name;
  pLastAnchor->param_short = param_short;
  pLastAnchor->param_propname = param_propname;
  pLastAnchor->min_val = min_val;
  pLastAnchor->max_val = max_val;
  pLastAnchor->defaultval = defaultval;
  pLastAnchor->defaultstr = defaultstr;
  pLastAnchor->paramrule = paramrule;
  pLastAnchor->proptyp.value = proptyp;
  pLastAnchor->stringlist = stringlist;
  pLastAnchor->group = grouptyp;
  pLastAnchor->memmap = memmap;

  return (BX_E_OK);
}


/* ------------------------------------------------------------------------
 *
 * This function initializes the dynamic capabilities
 * the pointer within the handle struct with the capabilities info
 * is filled and can get accessed afterwards.
 *
 * ------------------------------------------------------------------------ */
bx_errtype EXPORT BestXCapaInit(
    bx_handletype handle)
{
  bx_int32 lic;                  /* license code */
  bx_int32 i;             /* general purpose counter */
  bx_errtype __status;
  bx_int32 num_elem;             /* holds the number of elements for each
                  * parameter block */
  bx_param_infotype *p_anchor;   /* just a dummy variable to hold the param
                  * info */
  int zwcnt;
  bx_int32 tracememdepth;        /* short cut ... used often */

  bx_int32 GUIUsageCode;         /* Ichiban 2.0 limitation concerning
                    m_lock and t_lock
                                 */


  /* SCR; ptrs to elements of bx_handlearray[handle].capable */
  bx_dynamic_capability_type *pCapable = bx_handlearray[handle].capable = NULL;

  bx_generic_infotype *pRIBlockGen = NULL;

  bx_generic_infotype *pRIBehGen = NULL;
  bx_generic_infotype *pCTBehGen = NULL;
  bx_generic_infotype *pCIBehGen = NULL;
  bx_generic_infotype *pRTBehGen = NULL;
  bx_generic_infotype *pCTSplitCondGen = NULL;

  bx_generic_infotype *pRIGen = NULL;
  bx_generic_infotype *pExerciserGen = NULL;
  bx_generic_infotype *pAnalyzerGen = NULL;
  bx_generic_infotype *pCTGen = NULL;
  bx_generic_infotype *pCIGen = NULL;
  bx_generic_infotype *pRTGen = NULL;

  bx_generic_infotype *pDecoderGen = NULL;
  bx_generic_infotype *pDecoderStdGen = NULL;
  bx_generic_infotype *pDecoderExpGen = NULL;
  bx_generic_infotype *pDecoderReqGen = NULL;
  bx_generic_infotype *pDecoderCfgGen = NULL;
  bx_generic_infotype *pDatamemoryGen = NULL;
  bx_generic_infotype *pTraceGenGen = NULL;
  bx_generic_infotype *pTracememGen = NULL;
  bx_generic_infotype *pPatternGen = NULL;
  bx_generic_infotype *pSeqTriggerGenGen = NULL;
  bx_generic_infotype *pSeqTriggerTranGen = NULL;
  bx_generic_infotype *pSeqTriggerCondGen = NULL;
  bx_generic_infotype *pSeqPerforGenGen = NULL;
  bx_generic_infotype *pSeqPerforTranGen = NULL;
  bx_generic_infotype *pSeqPerforCondGen = NULL;
  bx_generic_infotype *pObserverRulesGen = NULL;
  bx_generic_infotype *pStatusGen = NULL;
#if 0
  bx_generic_infotype *pRIBehGroupInfoGen = NULL;
  bx_generic_infotype *pCTBehGroupInfoGen = NULL;
#endif
  bx_generic_infotype *pBoardGen = NULL;

  void * pVoid;                  /* resolves different addressing systems */
  bx_hwtype theBoard;

  /* but these 2 are the most important */
  bx_bool fIsDeep = BestXIsDeep(handle);
  bx_bool fIsCompact = BestXIsCompact(handle);

  BestXGetHWFromHandle(handle, &theBoard);
  
  if (NULL != pCapable)         /* must be zero here !! */
  {
    BX_ERRETURN(BX_E_DYNAMIC_CAPABILITY);
  }

  /* we can't do anything with unknown hardware */
  if(!BestXIsKnown(handle))
  {
    BestXLastErrorParamSet(handle, BX_ERRPAR_1,
               (bx_int32) bx_handlearray[handle].port);

    BX_ERRETURN(BX_E_UNKNOWN_HARDWARE);
  }

  /* the struct itself is allocated here */
  /* malloc, chk ptr and zero out new mem */
  BX_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(bx_dynamic_capability_type)));

  /* short cut */
  pCapable = (bx_dynamic_capability_type *) pVoid;
  bx_handlearray[handle].capable = pCapable;

  /* malloc, chk ptr and zero out new mem */
  BX_ERRCHECK(malloc_chk_zero(&pVoid, (size_t) (sizeof(bx_int32) * 3)));

  pCapable->capa_code = (bx_int32 *) pVoid;

  /* determine the license here ... first one only */
  BX_ERRCHECK(BestXIHWCapabilityGet(handle, pCapable->capa_code));
  lic = pCapable->capa_code[0]; /* the real license */


  /* -------------------------------------------------------------------- */
  /* -------------------------------------------------------------------- */
  /* ---------------Construct the dynamic Arrays------------------------ */
  /* -------------------------------------------------------------------- */
  /* -------------------------------------------------------------------- */

  /* initialize storage for the info struct -> param_last times */
  BX_ERRCHECK(malloc_chk_zero(&pVoid,
                  (size_t)(BX_PARAM_LAST) * sizeof (bx_infotype)));

  pCapable->info = (bx_infotype *) pVoid;

  for (i=0; i<BX_PARAM_LAST; i++)
  {
    /* initialize storage for generic info */
    /* param info is generated automatically */

    BX_ERRCHECK(malloc_chk_zero(&pVoid, sizeof (bx_generic_infotype)));
    pCapable->info[i].geninfo = (bx_generic_infotype *) pVoid;
  }

  /* ------------------------------------------------------------------------
   * Parameterslists follow here. The information
   * is organized as follows: Name, Short name, minimal value, maximal
   * value,default value,rule to follow, Property enum, optional stringlist,
   * [6] syntax infos, License needed. Syntax info contains the
   * optional/required switch for each of the commands
   * (m_block,m_xact,m_data,m_last,m_attr,t_attr) of the xact compiler.
   * -------------------------------------------------------------------- */

  /* -------------------------------------------------------------------- */
  /* RI block information (generic and the parameter itself) */
  /* -------------------------------------------------------------------- */

  /* short cut */
  pRIBlockGen = pCapable->info[BX_PARAM_RI_BLOCK].geninfo;

  /* group name */
  pRIBlockGen->group_name = "RI Block";

  /* number of RI block properties */
  pRIBlockGen->num_elem = 0;/* default */

  pRIBlockGen->depth = 256;

  pRIBlockGen->int_mem_offset = 65536L;
  pRIBlockGen->fixeddelay = 0;  /* no delay between blocks */

  /* the  RI block parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (1) /*lic & BX_CAPABILITY_EXERCISER ||
       lic & BX_CAPABILITY_HOSTINT)  this is the prerequisite */
  {
    ADDPAR(( "Bus Command", "Command",        /* name, short name */
         0x0UL, 0xdffdUL/* no special,DAC */, /* minval, maxval */
         BX_RIBLK_BUSCMD_MEM_READDWORD,      /* default */
         NULL,                   /* default str */
         BX_PARAMRULE_PPRVAR|BX_PARAMRULE_BITSET, /* rule */
             BX_RIBLK_BUSCMD,      /* property */
         BX_RIBLK_BUSCMD_table,          /* stringlist */
             BX_RIBLKGRP_3 , Mephisto_RI_Blk2Hw));

    ADDPAR(( "Bus Address Lo", "AD32",                     /* name, short name */
         0x0UL, 0xffffffffUL,        /* minval, maxval */
         0UL, NULL,                /* default, default str */
             BX_PARAMRULE_NORULE,        /* rule */
             BX_RIBLK_BUSADDR_LO,    /* property */
             NULL,                /* stringlist */
         BX_RIBLKGRP_0 , Mephisto_RI_Blk2Hw));

    ADDPAR(( "Bus Address Hi", "AD64",        /* name, short name */
         0x0UL, 0xffffffffL,        /* minval, maxval */
         0UL, NULL,                /* default, default str */
             BX_PARAMRULE_NORULE,        /* rule */
             BX_RIBLK_BUSADDR_HI,    /* property */
         NULL,                /* stringlist */
             BX_RIBLKGRP_1 , Mephisto_RI_Blk2Hw));

    /* Burst size in bytes */
    ADDPAR(( "Number of Bytes", "NumBytes",
         0x1UL, 0xffffffff,
         1UL, NULL,
         BX_PARAMRULE_PPRVAR,
             BX_RIBLK_NUMBYTES,
             NULL, BX_RIBLKGRP_2 , Mephisto_RI_Blk2Hw));

    /* Internal memory size Mephisto: 1MByte (128k * 64 bit) */
    /* Internal memory size Faust:    4MByte (512k * 64 bit) */
    ADDPAR(( "Internal Address", "IntAddr",
         0x0UL, (BestXHasMephisto(handle)?0xfffff:0x3fffff),
         0UL, NULL,
             BX_PARAMRULE_DWORDBOUND,
         BX_RIBLK_INTADDR,
         NULL,
             BX_RIBLKGRP_3 , Mephisto_RI_Blk2Hw));

    ADDPAR(( "Resource","Resource", 
         0x0UL, 0x1L,
         BX_RIBLK_RESOURCE_DATAMEM, NULL,
         BX_PARAMRULE_NORULE,
             BX_RIBLK_RESOURCE,
         BX_RIBLK_RESOURCE_table,
             BX_RIBLKGRP_4 , Mephisto_RI_Blk2Hw));

    ADDPAR(( "Data Compare", "DataCmp",
         0x0UL, 0x1L,
         BX_RIBLK_DATACMP_OFF, NULL,
         BX_PARAMRULE_NORULE,
             BX_RIBLK_DATACMP,
         onoff_table,
             BX_RIBLKGRP_4 , Mephisto_RI_Blk2Hw));

    ADDPAR(( "Conditional Start", "CondStart",
             BX_RIBLK_CONDSTART_ONCE1, BX_RIBLK_CONDSTART_WAIT2,
         BX_RIBLK_CONDSTART_NO, NULL,
         BX_PARAMRULE_ZEROALLOWED,
             BX_RIBLK_CONDSTART,
         BX_RIBLK_CONDSTART_table,
             BX_RIBLKGRP_3,
         Mephisto_RI_Blk2Hw));

    ADDPAR(( "Byte Enables", "Byten",
         0x0UL, 0xf,
         0UL, NULL,
         BX_PARAMRULE_PPRVAR,
             BX_RIBLK_BYTEN,
         BX_RIBLK_BYTEN_table,
             BX_RIBLKGRP_3,
         Mephisto_RI_Blk2Hw));

    ADDPAR(( "Completion", "Completion",
         0x0UL, 0x1UL,
         0x0UL, NULL,
         BX_PARAMRULE_NORULE,
             BX_RIBLK_COMPLETION, yesno_table,
             BX_RIBLKGRP_3,
         Mephisto_RI_Blk2Hw));

    ADDPAR(( "Queue", "Queue",
             1UL, 3UL,
         BX_RIBLK_QUEUE_A, NULL,
         BX_PARAMRULE_NORULE,
             BX_RIBLK_QUEUE,
         BX_RIBLK_QUEUE_table,
             BX_RIBLKGRP_4 , Mephisto_RI_Blk2Hw));

    ADDPAR(( "Relaxed Ordering", "RelaxOrder",
         0x0UL, 0x1UL,
         1UL, NULL,
         BX_PARAMRULE_PPRVAR,
             BX_RIBLK_RELAXORDER,
         yesno_table,
             BX_RIBLKGRP_4 , Mephisto_RI_Blk2Hw));

    ADDPAR(( "No Snoop", "NoSnoop",
         0x0UL, 0x1UL,
         0UL, NULL,
         BX_PARAMRULE_PPRVAR,
             BX_RIBLK_NOSNOOP,
         yesno_table,
             BX_RIBLKGRP_4 , Mephisto_RI_Blk2Hw));

    ADDPAR(( "Reserved31", "Reser31",
         0x0UL, 0x1UL,
         0UL, NULL,
         BX_PARAMRULE_NORULE,
             BX_RIBLK_RESERVED31,
             NULL,
         BX_RIBLKGRP_4 , Mephisto_RI_Blk2Hw));


    /* associate the right number of elements */
    pRIBlockGen->num_elem = num_elem;
  }

  pCapable->info[BX_PARAM_RI_BLOCK].paraminfo = p_anchor; /* set it */


  /* -------------------------------------------------------------------- */
  /* the RI behavior information (generic and parameter) */
  /* this is the only place where we handle the loop grouping of the */
  /* parameters (except target attributes) */
  /* for every parameter there is a loop bit that is reponsible for it */
  /* to enable ppr to access this information, the structure holds the */
  /* info. we do distinguish between e2925a and e2926a in this list */
  /* -------------------------------------------------------------------- */

  /* short cut */
  pRIBehGen = pCapable->info[BX_PARAM_RI_BEH].geninfo;

  pRIBehGen->group_name = "RI Behavior";  /* group name */

  /* Caution: Because the attributepagesize is variable in Ichiban boards,
   * the following values describe only the maximum values.
   * The exact range checking is done in the firmware, not in the CAPI. */
  pRIBehGen->depth = 256;          /* depth of the memory */
  pRIBehGen->int_mem_offset = 65536L;  /* internal data mem offset
                                        * (xact) */
  /* the RI beh parameter information */
  /* set to NULL to make AddParam Fct start with malloc */
  p_anchor = NULL;
  num_elem = 0;

  if (1) /*lic & BX_CAPABILITY_EXERCISER) */
  {
    ADDPAR(( "Queue", "Queue",
         1, 3,
         BX_RIBEH_QUEUE_A, NULL,
         BX_PARAMRULE_PPRVAR,
             BX_RIBEH_QUEUE,
         BX_RIBEH_QUEUE_table,
             BX_RIBEHGRP_6 , Mephisto_RI_Beh2Hw));

    ADDPAR(( "Tag","Tag",
             0UL, 32UL,
         BX_RIBEH_TAG_AUTO, NULL,
             BX_PARAMRULE_NORULE,
             BX_RIBEH_TAG, BX_RIBEH_TAG_table,
             BX_RIBEHGRP_7 , Mephisto_RI_Beh2Hw));

    ADDPAR(( "Byte Count", "ByteCount",
             1UL, 4096UL,
         4096UL, NULL,
         BX_PARAMRULE_PPRVAR,
             BX_RIBEH_BYTECOUNT,
         NULL,
             BX_RIBEHGRP_8 | BX_RIBEHGRP_9 , Mephisto_RI_Beh2Hw));

    ADDPAR(( "Disconnect", "Disconnect", 
             0UL, 32UL,
         BX_RIBEH_DISCONNECT_NO, NULL,
         BX_PARAMRULE_PPRVAR,
         BX_RIBEH_DISCONNECT,
             BX_RIBEH_DISCONNECT_table,
             BX_RIBEHGRP_3 , Mephisto_RI_Beh2Hw));


    ADDPAR(( "Delay","Delay",
         1UL, 65535UL,
         1UL, NULL,
         BX_PARAMRULE_PPRVAR,
             BX_RIBEH_DELAY,
         NULL,
             BX_RIBEHGRP_2 | BX_RIBEHGRP_4 | BX_RIBEHGRP_5 , Mephisto_RI_Beh2Hw));

    ADDPAR(( "Steps","Steps",
         0x0UL, 0x4UL /* chris TBD: 6 for non-compliant behavior */,
         0UL, NULL,
         BX_PARAMRULE_PPRVAR,
             BX_RIBEH_STEPS,NULL,
             BX_RIBEHGRP_6 , Mephisto_RI_Beh2Hw));

    ADDPAR(( "Req64","Req64",
         0x0UL, 0x1UL,
         1UL, NULL,
         BX_PARAMRULE_PPRVAR,
         BX_RIBEH_REQ64,
         yesno_table,
         BX_RIBEHGRP_6 , Mephisto_RI_Beh2Hw));

    ADDPAR(( "Release REQ#","RelReq",
             0UL /* 1 ?? chris TBD */, 2047UL,
         2047UL, NULL,
         BX_PARAMRULE_PPRVAR,
             BX_RIBEH_RELREQ,
         NULL /* no table */,
             BX_RIBEHGRP_1 | BX_RIBEHGRP_2 , Mephisto_RI_Beh2Hw));

    ADDPAR(( "Repeat","Repeat",
         0x1UL, 256UL,
         1UL, NULL,
         BX_PARAMRULE_NORULE,
             BX_RIBEH_REPEAT,
         NULL /* no table */,
             BX_RIBEHGRP_0 , Mephisto_RI_Beh2Hw));

    ADDPAR(( "Skip","Skip",
         0x0UL, 7UL,
         BX_RIBEH_SKIP_NO, NULL,
         BX_PARAMRULE_NORULE,
             BX_RIBEH_SKIP, BX_RIBEH_SKIP_table,
             BX_RIBEHGRP_6 | BX_RIBEHGRP_7 | BX_RIBEHGRP_9,
         Mephisto_RI_Beh2Hw));

  }

  /* associate the right number of elements */
  pRIBehGen->num_elem = num_elem;
  pCapable->info[BX_PARAM_RI_BEH].paraminfo = p_anchor;  /* set it */


  /* -------------------------------------------------------------------- */
  /* the RI generic property information (generic and parameters) */
  /* -------------------------------------------------------------------- */

  /* short cut */
  pRIGen = pCapable->info[BX_PARAM_RI_GEN].geninfo;

  /* leave depth of the RI gen memory at 0 (there is no such memory) */
  pRIGen->group_name = "RI Generic";

  /* the RI parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (1) /*lic & BX_CAPABILITY_EXERCISER) */
  {
    ADDPAR(( "BX_RIGEN_MENABLE", NULL, 0x0UL, 1UL, 0UL, NULL, 0UL,
             BX_RIGEN_MENABLE,
             NULL, 0UL, NULL));

    ADDPAR(( "BX_RIGEN_NUMBLK", NULL, 0x1UL, 256UL, 1UL, NULL, 0UL,
             BX_RIGEN_NUMBLK,
             NULL, 0UL, NULL));

    ADDPAR(( "BX_RIGEN_REPEATBLK", NULL, 0x0UL, 0xffffffffUL, 1UL, NULL, 0UL,
             BX_RIGEN_REPEATBLK, BX_RIGEN_REPEATBLK_table,
             0UL, NULL));

    ADDPAR(( "BX_RIGEN_NUMBEH", NULL, 0x1UL, 256UL, 1UL, NULL, 0UL,
             BX_RIGEN_NUMBEH,
             NULL, 0UL , NULL));

    ADDPAR(( "BX_RIGEN_NUMBEHG0", NULL, 0x1UL, 256UL, 1UL, NULL, 0UL,
             BX_RIGEN_NUMBEHG0,
             NULL, BX_RIBEHGRP_0 , NULL));

    ADDPAR(( "BX_RIGEN_NUMBEHG1", NULL, 0x1UL, 256UL, 1UL, NULL, 0UL,
             BX_RIGEN_NUMBEHG1,
             NULL, BX_RIBEHGRP_1 , NULL));

    ADDPAR(( "BX_RIGEN_NUMBEHG2", NULL, 0x1UL, 256UL, 1UL, NULL, 0UL,
             BX_RIGEN_NUMBEHG2,
             NULL, BX_RIBEHGRP_2 , NULL));

    ADDPAR(( "BX_RIGEN_NUMBEHG3", NULL, 0x1UL, 256UL, 1UL, NULL, 0UL,
             BX_RIGEN_NUMBEHG3,
             NULL, BX_RIBEHGRP_3 , NULL));

    ADDPAR(( "BX_RIGEN_NUMBEHG4", NULL, 0x1UL, 256UL, 1UL, NULL, 0UL,
             BX_RIGEN_NUMBEHG4,
             NULL, BX_RIBEHGRP_4 , NULL));

    ADDPAR(( "BX_RIGEN_NUMBEHG5", NULL, 0x1UL, 256UL, 1UL, NULL, 0UL,
             BX_RIGEN_NUMBEHG5,
             NULL, BX_RIBEHGRP_5 , NULL));

    ADDPAR(( "BX_RIGEN_NUMBEHG6", NULL, 0x1UL, 256UL, 1UL, NULL, 0UL,
             BX_RIGEN_NUMBEHG6,
             NULL, BX_RIBEHGRP_6 , NULL));

    ADDPAR(( "BX_RIGEN_NUMBEHG7", NULL, 0x1UL, 256UL, 1UL, NULL, 0UL,
             BX_RIGEN_NUMBEHG7,
             NULL, BX_RIBEHGRP_7 , NULL));

    ADDPAR(( "BX_RIGEN_NUMBEHG8", NULL, 0x1UL, 256UL, 1UL, NULL, 0UL,
             BX_RIGEN_NUMBEHG8,
             NULL, BX_RIBEHGRP_8 , NULL));

    ADDPAR(( "BX_RIGEN_NUMBEHG9", NULL, 0x1UL, 256UL, 1UL, NULL, 0UL,
             BX_RIGEN_NUMBEHG9,
             NULL, BX_RIBEHGRP_9 , NULL));

    /* chris: TBD: Skip values must be qword-aligned !! */

    ADDPAR(( "BX_RIGEN_SKIPREG1", NULL, 0x0UL, 8192UL, 0UL, NULL, 0UL,
             BX_RIGEN_SKIPREG1,
             NULL, 0UL, NULL));

    ADDPAR(( "BX_RIGEN_SKIPREG2", NULL, 0x0UL, 8192UL, 0UL, NULL, 0UL,
             BX_RIGEN_SKIPREG2,
             NULL, 0UL, NULL));

    ADDPAR(( "BX_RIGEN_SKIPREG3", NULL, 0x0UL, 8192UL, 0UL, NULL, 0UL,
             BX_RIGEN_SKIPREG3,
             NULL, 0UL, NULL));

    ADDPAR(( "BX_RIGEN_SKIPREG4", NULL, 0x0UL, 8192UL, 0UL, NULL, 0UL,
             BX_RIGEN_SKIPREG4,
             NULL, 0UL, NULL));

    ADDPAR(( "BX_RIGEN_SKIPREG5", NULL, 0x0UL, 8192UL, 0UL, NULL, 0UL,
             BX_RIGEN_SKIPREG5,
             NULL, 0UL, NULL));

    ADDPAR(( "BX_RIGEN_SKIPREG6", NULL, 0x0UL, 8192UL, 0UL, NULL, 0UL,
             BX_RIGEN_SKIPREG6,
             NULL, 0UL, NULL));

    ADDPAR(( "BX_RIGEN_SKIPREG7", NULL, 0x0UL, 8192UL, 0UL, NULL, 0UL,
             BX_RIGEN_SKIPREG7,
             NULL, 0UL, NULL));

    ADDPAR(( "BX_RIGEN_TABORT", NULL, 0x0UL, 0x1UL,BX_RIGEN_TABORT_STOP , NULL, 0UL,
             BX_RIGEN_TABORT, BX_RIGEN_TABORT_table,
             0UL, NULL));

    ADDPAR(( "BX_RIGEN_IABORT", NULL, 0x0UL, 0x1UL,BX_RIGEN_IABORT_STOP , NULL, 0UL,
             BX_RIGEN_IABORT, BX_RIGEN_IABORT_table,
             0UL, NULL));

  }

  /* associate the right number of elements */
  pRIGen->num_elem = num_elem;

  pCapable->info[BX_PARAM_RI_GEN].paraminfo = p_anchor; /* set it */

  /* -------------------------------------------------------------------- */
  /* the exercisergenericproperty information (generic and parameters) */
  /* -------------------------------------------------------------------- */

  /* short cut */
  pExerciserGen = pCapable->info[BX_PARAM_EXERCISER_GEN].geninfo;

  /* leave depth of the RI gen memory at 0 (there is no such memory) */
  pExerciserGen->group_name = "Exerciser Generic";

  /* the exerciser generic parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;


  if (!BestXHasMephisto(handle))
  {
    ADDPAR(( "BX_EGEN_ERR_ECC","errecc", 0x0UL, 0x2UL, 0UL, NULL, 0UL,
         BX_EGEN_ERR_ECC,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_EGEN_ERR_SUBPHASE","errsubphase", 0x0UL, 0xfUL, 1UL, NULL, 0UL,
         BX_EGEN_ERR_SUBPHASE,
         NULL, 0UL, NULL));
  }

  ADDPAR(( "BX_EGEN_COMPLIANT","compliant", 0x0UL, 0x1UL, 1UL, NULL, 0UL,
       BX_EGEN_COMPLIANT,
       NULL, 0UL, NULL));


  ADDPAR(( "BX_EGEN_ERR_SOURCE","errsource", 0x0UL, 0x6UL, BX_EGEN_ERR_SOURCE_NONE, NULL, 0UL,
       BX_EGEN_ERR_SOURCE, BX_EGEN_ERR_SOURCE_table,
       0UL, NULL));

  /* This range of this one is dependent on BX_EGEN_ERR_SOURCE !! */
  ADDPAR(( "BX_EGEN_ERR_NUM","errnum", 0x0UL, 255UL, 0UL, NULL, 0UL,
       BX_EGEN_ERR_NUM,
       NULL, 0UL, NULL));

  ADDPAR(( "BX_EGEN_ERR_PHASE","errphase", 0x0UL /*ADDR*/, 1025UL /*ATTR*/,
       BX_EGEN_ERR_PHASE_ADDR, NULL, 0UL,
       BX_EGEN_ERR_PHASE, BX_EGEN_ERR_PHASE_table,
       0UL, NULL));

  ADDPAR(( "BX_EGEN_ERR_WRPAR","wrpar", 0x0UL, 0x1UL, 0UL, NULL, 0UL,
       BX_EGEN_ERR_WRPAR,
       NULL, 0UL, NULL));

  ADDPAR(( "BX_EGEN_ERR_WRPAR64","wrpar64", 0x0UL, 0x1UL, 0UL, NULL, 0UL,
       BX_EGEN_ERR_WRPAR64,
       NULL, 0UL, NULL));

  ADDPAR(( "BX_EGEN_ERR_PERR","perr", 0x0UL, 0x1UL, 0UL, NULL, 0UL,
       BX_EGEN_ERR_PERR,
       NULL, 0UL, NULL));

  ADDPAR(( "BX_EGEN_ERR_SERR","serr", 0x0UL, 0x1UL, 0UL, NULL, 0UL,
       BX_EGEN_ERR_SERR,
       NULL, 0UL, NULL));

  ADDPAR(( "BX_EGEN_INT_SOURCE","intsource", 0x0UL, (BestXHasFaust(handle)?7UL:6UL), BX_EGEN_INT_SOURCE_NONE, NULL, 0UL,
       BX_EGEN_INT_SOURCE, BX_EGEN_INT_SOURCE_table,
       0UL, NULL));

  /* This range of this one is dependent on BX_EGEN_INT_SOURCE !!
     Value zero is needed for decoder BAR0 as interrupt source !!  */
  ADDPAR(( "BX_EGEN_INT_NUM","intnum", 0x0UL, 255UL, 0UL, NULL, 0UL,
       BX_EGEN_INT_NUM,
       NULL, 0UL, NULL));

  ADDPAR(( "BX_EGEN_INT_DELAYA","intdelaya", 0x0UL, 65535UL, BX_EGEN_INT_DELAY_NO, NULL, 0UL,
       BX_EGEN_INT_DELAYA,
       NULL, 0UL, NULL));

  ADDPAR(( "BX_EGEN_INT_DELAYB","intdelayb", 0x0UL, 65535UL, BX_EGEN_INT_DELAY_NO, NULL, 0UL,
       BX_EGEN_INT_DELAYB,
       NULL, 0UL, NULL));

  ADDPAR(( "BX_EGEN_INT_DELAYC","intdelayc", 0x0UL, 65535UL, BX_EGEN_INT_DELAY_NO, NULL, 0UL,
       BX_EGEN_INT_DELAYC,
       NULL, 0UL, NULL));

  ADDPAR(( "BX_EGEN_INT_DELAYD","intdelayd", 0x0UL, 65535UL, BX_EGEN_INT_DELAY_NO, NULL, 0UL,
       BX_EGEN_INT_DELAYD,
       NULL, 0UL, NULL));

  ADDPAR(( "BX_EGEN_TRIG_SOURCE","trigsource", 0x0UL, 0x6UL, 0UL, NULL, 0UL,
       BX_EGEN_TRIG_SOURCE, BX_EGEN_TRIG_SOURCE_table,
       0UL, NULL));

  /* This range of this one is dependent on BX_EGEN_TRIG_SOURCE !!  */
  ADDPAR(( "BX_EGEN_TRIG_NUM","trignum", 0x0UL, 255UL, 0UL, NULL, 0UL,
       BX_EGEN_TRIG_NUM,
       NULL, 0UL, NULL));

  ADDPAR(( "BX_EGEN_ARB","arb", 0, 3, BX_EGEN_ARB_AUTO, NULL, 0UL,
       BX_EGEN_ARB, BX_EGEN_ARB_table,
       0UL, NULL));


  /* Caution: In increment-mode the min-registers in HW
     (i.e. ARBRI and ARBCI) always have to be LESS THAN the
     max-registers (set fix to 255).  For this we allow these two
     props not to exceed a value of 254 (chris).  */
  ADDPAR(( "BX_EGEN_ARBRI","arbri", 1, 254, 1, NULL, 0UL,
       BX_EGEN_ARBRI,
       NULL, 0UL, NULL));

  /* Caution: In increment-mode the min-registers in HW (i.e. ARBRI and ARBCI)
     always have to be LESS THAN the max-registers (set fix to 255).
     For this we allow these two props not to exceed a value of 254 (chris).
  */
  ADDPAR(("BX_EGEN_ARBCI","arbci", 1, 254, 1, NULL, 0UL,
      BX_EGEN_ARBCI,
      NULL, 0UL, NULL));

  ADDPAR(( "BX_EGEN_DATAGEN","datagen", 0,4, BX_EGEN_DATAGEN_COUNTER, NULL, 0UL,
       BX_EGEN_DATAGEN, BX_EGEN_DATAGEN_table,
       0UL, NULL));

  ADDPAR(( "BX_EGEN_DATAFIX","datafix",
       0, BX_EGEN_DATAFIX_MASTERID,BX_EGEN_DATAFIX_MASTERID,
       NULL, 0UL,
       BX_EGEN_DATAFIX,
       NULL, 0UL, NULL));

  ADDPAR(( "BX_EGEN_DATASEED","dataseed", 0, 0xfffff, 0, NULL, 0UL,
       BX_EGEN_DATASEED,
       NULL, 0UL, NULL));

  ADDPAR(("BX_EGEN_PARTCOMP","partcomp", 0, 1, BX_EGEN_PARTCOMP_OFF, NULL, 0UL,
      BX_EGEN_PARTCOMP,
      NULL, 0UL, NULL));


  /* associate the right number of elements */
  pExerciserGen->num_elem = num_elem;

  pCapable->info[BX_PARAM_EXERCISER_GEN].paraminfo = p_anchor;  /* set it */

  /* -------------------------------------------------------------------- */
  /* the analyzer genericproperty information (generic and parameters) */
  /* -------------------------------------------------------------------- */

  /* short cut */
  pAnalyzerGen = pCapable->info[BX_PARAM_ANALYZER_GEN].geninfo;

  pAnalyzerGen->group_name = "Analyzer Generic";

  /* the analyzer generic parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  /* 12 bit HW register;
     A value of zero in HW (HW default: 0xfff) means 0x1000
     A value of one would trigger the rule immediately 
     
   * is organized as follows: Name, Short name, minimal value, maximal
   * value,default value,rule to follow, Property enum, optional stringlist,
   * [6] syntax infos, License needed. Syntax info contains the
   * optional/required switch for each of the commands
  */
  
  ADDPAR(( "Semantic 10", "sem10", 0x0UL, 0xfffUL, 0xfffUL /* HW default */, NULL, 0UL,
       BX_AGEN_SEM10,
       NULL, 0UL, NULL));

  /* 20 bit HW register;
     A value of zero in HW (HW default !) means 0x100000
     A value of one would trigger the rule immediately */
  ADDPAR(( "Semantic 11", "sem11", 0x0UL, 0xfffffUL, 0x0UL /* HW default */, NULL, 0UL,
       BX_AGEN_SEM11,
       NULL, 0UL, NULL));


  /* associate the right number of elements */
  pAnalyzerGen->num_elem = num_elem;

  pCapable->info[BX_PARAM_ANALYZER_GEN].paraminfo = p_anchor;  /* set it */


  /* -------------------------------------------------------------------- */
  /* the targetbehaviorproptery information (generic and parameters)      */
  /* -------------------------------------------------------------------- */

  /* short cut */
  pCTBehGen = pCapable->info[BX_PARAM_CT_BEH].geninfo;

  /* group name */
  pCTBehGen->group_name = "Completer Target Behavior";

  pCTBehGen->depth = 256;

  /* the targetbeh parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;
  if (1) /* lic & BX_CAPABILITY_EXERCISER) */
  {
    ADDPAR(( "Decode Speed","DecSpeed",
         BX_CTBEH_DECSPEED_A, BX_CTBEH_DECSPEED_C,
         BX_CTBEH_DECSPEED_B, NULL,
         BX_PARAMRULE_PPRVAR,
         BX_CTBEH_DECSPEED,
         BX_CTBEH_DECSPEED_table,
         BX_CTBEHGRP_4, Mephisto_CT_Beh2Hw));

    ADDPAR(( "Ack64","Ack64",
         0x0UL, 0x1UL,
         1UL, NULL,
         BX_PARAMRULE_PPRVAR,
         BX_CTBEH_ACK64,
         yesno_table,
         BX_CTBEHGRP_4, Mephisto_CT_Beh2Hw));

    ADDPAR(( "Initial","Initial",
         0x0UL, 0x3UL,
         BX_CTBEH_INITIAL_ACCEPT, NULL,
         BX_PARAMRULE_PPRVAR,
         BX_CTBEH_INITIAL,
         BX_CTBEH_INITIAL_table,
         BX_CTBEHGRP_1, Mephisto_CT_Beh2Hw));

    ADDPAR(( "Latency","Latency",
         0x3UL, 34UL,
         3UL, NULL,
         BX_PARAMRULE_PPRVAR,
         BX_CTBEH_LATENCY,
         NULL /* no table */,
         BX_CTBEHGRP_1, Mephisto_CT_Beh2Hw));

    ADDPAR(( "Subseq","Subseq",
         0x0UL, (BestXHasMephisto(handle)?0x1:0x2),
         BX_CTBEH_SUBSEQ_ACCEPT , NULL,
         BX_PARAMRULE_PPRVAR,
         BX_CTBEH_SUBSEQ,
         BX_CTBEH_SUBSEQ_table,
         BX_CTBEHGRP_3, Mephisto_CT_Beh2Hw));

    ADDPAR(( "Subseqphase","SubseqPhase",
         0x0UL, 2047UL,
         0UL, NULL,
         BX_PARAMRULE_PPRVAR,
         BX_CTBEH_SUBSEQPHASE, NULL /* notable */,
         BX_CTBEHGRP_2 | BX_CTBEHGRP_3, Mephisto_CT_Beh2Hw));

    ADDPAR(( "Split Latency","SplitLatency",
         3UL, 18UL,
         3UL, NULL,
         BX_PARAMRULE_PPRVAR,
         BX_CTBEH_SPLITLATENCY,
         NULL /* no table */,
         BX_CTBEHGRP_4, Mephisto_CT_Beh2Hw));

    ADDPAR(( "Split Enable","SplitEnable",
         0UL, 1UL,
         1UL, NULL,
         BX_PARAMRULE_PPRVAR,
         BX_CTBEH_SPLITENABLE,
         yesno_table,
         BX_CTBEHGRP_3, Mephisto_CT_Beh2Hw));

    /* Has special handling in xct.c */
    ADDPAR(( "Repeat","Repeat", 
         0x1UL, 0x10000UL,
         1UL, NULL,
         BX_PARAMRULE_NORULE,
         BX_CTBEH_REPEAT,
         NULL /* no table */,
         BX_CTBEHGRP_0, Mephisto_CT_Beh2Hw));
  
  }

  /* associate the right number of elements */
  pCTBehGen->num_elem = num_elem;

  pCapable->info[BX_PARAM_CT_BEH].paraminfo = p_anchor;  /* set it */

  /* -------------------------------------------------------------------- */
  /* the targetgenericproperty information (generic and parameters) */
  /* -------------------------------------------------------------------- */

  /* short cut */
  pCTGen = pCapable->info[BX_PARAM_CT_GEN].geninfo;

  /* leave depth of the target gen memory = 0 (there is no such memory) */
  pCTGen->group_name = "Completer Target Generic";

  /* the targetgen parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (1) /* lic & BX_CAPABILITY_EXERCISER ||
            lic & BX_CAPABILITY_HOSTINT) */
  {
    ADDPAR(( "BX_CTGEN_NUMBEH", NULL, 1UL, 256UL,1UL,
         NULL, 0UL, BX_CTGEN_NUMBEH,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_CTGEN_NUMBEHG0", NULL, 1UL, 256UL,1UL,
         NULL, 0UL, BX_CTGEN_NUMBEHG0,
         NULL, BX_CTBEHGRP_0 , NULL));

    ADDPAR(( "BX_CTGEN_NUMBEHG1", NULL, 1UL, 256UL,1UL,
         NULL, 0UL, BX_CTGEN_NUMBEHG1,
         NULL, BX_CTBEHGRP_1 , NULL));

    ADDPAR(( "BX_CTGEN_NUMBEHG2", NULL, 1UL, 256UL,1UL,
         NULL, 0UL, BX_CTGEN_NUMBEHG2,
         NULL, BX_CTBEHGRP_2 , NULL));

    ADDPAR(( "BX_CTGEN_NUMBEHG3", NULL, 1UL, 256UL,1UL,
         NULL, 0UL, BX_CTGEN_NUMBEHG3,
         NULL, BX_CTBEHGRP_3 , NULL));

    ADDPAR(( "BX_CTGEN_NUMBEHG4", NULL, 1UL, 256UL,1UL,
         NULL, 0UL, BX_CTGEN_NUMBEHG4,
         NULL, BX_CTBEHGRP_4 , NULL));
  }

  /* associate the right number of elements */
  pCTGen->num_elem = num_elem;

  pCapable->info[BX_PARAM_CT_GEN].paraminfo = p_anchor; /* set it */

  /* -------------------------------------------------------------------- */
  /* the target split conditions information (generic and parameters)     */
  /* -------------------------------------------------------------------- */

  /* short cut */
  pCTSplitCondGen = pCapable->info[BX_PARAM_CT_SPLITCOND].geninfo;

  /* group name */
  pCTSplitCondGen->group_name = "Target Split Conditions";

  pCTSplitCondGen->depth = 256;

  /* the target split cond parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  /* In order to get a Spilt response, all these properties must be 'true', i.e. they are ANDed */
  /* The defaults are set identical here, so that each decoder splits always,
     but be aware that the final defaults are set/modified in BestXCTSplitCondDefaultSet() */
  
  /* HW limitation: The ADDRVAL and ADDRMASK do not work in E2930B.
     Mask=0 => Dont Care (always true)
     Mask=1 => Address on bus must match value */
  ADDPAR(( "Addrmask hi", "addrmask_hi", 0x0UL, 0xffffffffUL,0x0UL,
        NULL, 0UL, BX_CTSPLIT_ADDRMASK_HI,
        NULL, 0UL, NULL));

  ADDPAR(( "Addrmask_lo", "addrmask_lo", 0x0UL, 0xffffffffUL,0x0UL,
        NULL, 0UL, BX_CTSPLIT_ADDRMASK_LO,
        NULL, 0UL, NULL));

  ADDPAR(( "Addrval_hi", "addrval_hi", 0x0UL, 0xffffffffUL,0x0UL,
        NULL, 0UL, BX_CTSPLIT_ADDRVAL_HI,
        NULL, 0UL, NULL));

  ADDPAR(( "Addrval_lo", "addrval_lo", 0x0UL, 0xffffffffUL,0x0UL,
        NULL, 0UL, BX_CTSPLIT_ADDRVAL_LO,
        NULL, 0UL, NULL));

  ADDPAR(( "Cmds", "cmds", 0x0UL, 0xffffUL,BX_CTSPLIT_CMDS_ALL,
        NULL, 0UL, BX_CTSPLIT_CMDS, BX_CTSPLIT_CMDS_table,
        0UL, NULL));

  ADDPAR(( "Dec", "dec",
        0x0UL, BX_CTSPLIT_DEC_ANY, BX_CTSPLIT_DEC_ANY,
        NULL, 0UL,
        BX_CTSPLIT_DEC,
        BX_CTSPLIT_DEC_table,
        0UL, NULL));

  ADDPAR(( "Queue", "queue", 0x0UL, 0x11f /* i.e. 0 + {0,1,2,3,4,8} */,
        BX_CTSPLIT_QUEUE_AUTO,NULL, BX_PARAMRULE_BITSET,
        BX_CTSPLIT_QUEUE, BX_CTSPLIT_QUEUE_table,
        0UL, NULL));

  /* associate the right number of elements */
  pCTSplitCondGen->num_elem = num_elem;

  pCapable->info[BX_PARAM_CT_SPLITCOND].paraminfo = p_anchor;  /* set it */

  /* -------------------------------------------------------------------- */
  /* the completer generic property information (generic and parameters) */
  /* -------------------------------------------------------------------- */

  /* short cut */
  pCIGen = pCapable->info[BX_PARAM_CI_GEN].geninfo;

  /* leave depth of the target gen memory = 0 (there is no such memory) */
  pCIGen->group_name = "Completer Initiator Generic";

  /* the targetgen parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (1) /* lic & BX_CAPABILITY_EXERCISER ||
            lic & BX_CAPABILITY_HOSTINT) */
  {
    ADDPAR(( "BX_CIGEN_NUMBEH", NULL, 1UL, 256UL,1UL,
         NULL, 0UL, BX_CIGEN_NUMBEH,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_CIGEN_NUMBEHG0", NULL, 1UL, 256UL,1UL,
         NULL, 0UL, BX_CIGEN_NUMBEHG0,
         NULL, BX_CIBEHGRP_0 , NULL));

    ADDPAR(( "BX_CIGEN_NUMBEHG1", NULL, 1UL, 256UL,1UL,
         NULL, 0UL, BX_CIGEN_NUMBEHG1,
         NULL, BX_CIBEHGRP_1 , NULL));

    ADDPAR(( "BX_CIGEN_NUMBEHG2", NULL, 1UL, 256UL,1UL,
         NULL, 0UL, BX_CIGEN_NUMBEHG2,
         NULL, BX_CIBEHGRP_2 , NULL));

    ADDPAR(( "BX_CIGEN_NUMBEHG3", NULL, 1UL, 256UL,1UL,
         NULL, 0UL, BX_CIGEN_NUMBEHG3,
         NULL, BX_CIBEHGRP_3 , NULL));

    ADDPAR(( "BX_CIGEN_NUMBEHG4", NULL, 1UL, 256UL,1UL,
         NULL, 0UL, BX_CIGEN_NUMBEHG4,
         NULL, BX_CIBEHGRP_4 , NULL));

    ADDPAR(( "BX_CIGEN_NUMBEHG5", NULL, 1UL, 256UL,1UL,
         NULL, 0UL, BX_CIGEN_NUMBEHG5,
         NULL, BX_CIBEHGRP_5 , NULL));

    ADDPAR(( "BX_CIGEN_NUMBEHG6", NULL, 1UL, 256UL,1UL,
         NULL, 0UL, BX_CIGEN_NUMBEHG6,
         NULL, BX_CIBEHGRP_6 , NULL));

    ADDPAR(( "BX_CIGEN_NUMBEHG7", NULL, 1UL, 256UL,1UL,
         NULL, 0UL, BX_CIGEN_NUMBEHG7,
         NULL, BX_CIBEHGRP_7 , NULL));

    ADDPAR(( "BX_CIGEN_NUMBEHG8", NULL, 1UL, 256UL,1UL,
         NULL, 0UL, BX_CIGEN_NUMBEHG8,
         NULL, BX_CIBEHGRP_8 , NULL));

    ADDPAR(( "BX_CIGEN_MESSAGE_AD", NULL, 0x0UL, 0xffffffffUL,0UL,
         NULL, 0UL, BX_CIGEN_MESSAGE_AD,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_CIGEN_ADDR7", NULL, 0x0UL, 0x1UL,0UL,
         NULL, 0UL, BX_CIGEN_ADDR7,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_CIGEN_ADDR30", NULL, 0x0UL, 0x1UL,0UL,
         NULL, 0UL, BX_CIGEN_ADDR30,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_CIGEN_ADDR31", NULL, 0x0UL, 0x1UL,0UL,
         NULL, 0UL, BX_CIGEN_ADDR31,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_CIGEN_ATTR24", NULL, 0x0UL, 0x1UL,0UL,
         NULL, 0UL, BX_CIGEN_ATTR24,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_CIGEN_ATTR25", NULL, 0x0UL, 0x1UL,0UL,
         NULL, 0UL, BX_CIGEN_ATTR25,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_CIGEN_ATTR26", NULL, 0x0UL, 0x1UL,0UL,
         NULL, 0UL, BX_CIGEN_ATTR26,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_CIGEN_ATTR27", NULL, 0x0UL, 0x1UL,0UL,
         NULL, 0UL, BX_CIGEN_ATTR27,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_CIGEN_ATTR28", NULL, 0x0UL, 0x1UL,0UL,
         NULL, 0UL, BX_CIGEN_ATTR28,
         NULL, 0UL, NULL));
  }

  /* associate the right number of elements */
  pCIGen->num_elem = num_elem;

  pCapable->info[BX_PARAM_CI_GEN].paraminfo = p_anchor; /* set it */

  /* -------------------------------------------------------------------- */
  /* the completer behavior proptery information (generic and parameters)      */
  /* -------------------------------------------------------------------- */

  /* short cut */
  pCIBehGen = pCapable->info[BX_PARAM_CI_BEH].geninfo;

  /* group name */
  pCIBehGen->group_name = "Completer-Initiator Behavior";

  pCIBehGen->depth = 256;

  /* the targetbeh parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;
  if (1) /* lic & BX_CAPABILITY_EXERCISER) */
  {
    ADDPAR(( "Queue","Queue",
         BX_CIBEH_QUEUE_NEXT, BX_CIBEH_QUEUE_DWAIT,
         BX_CIBEH_QUEUE_NEXT, NULL,
         BX_PARAMRULE_PPRVAR,
         BX_CIBEH_QUEUE,
         BX_CIBEH_QUEUE_table,
         BX_CIBEHGRP_8, Mephisto_CI_Beh2Hw));

    ADDPAR(( "Addr7","Addr7",
         0UL, 0x1UL,
         0, NULL,
         BX_PARAMRULE_NORULE,
         BX_CIBEH_ADDR7, yesno_table,
         BX_CIBEHGRP_6, Mephisto_CI_Beh2Hw));

    ADDPAR(( "Addr30","Addr30", 0UL, 0x1UL, 0, NULL, 0UL,
         BX_CIBEH_ADDR30, yesno_table,
         BX_CIBEHGRP_5, Mephisto_CI_Beh2Hw));

    ADDPAR(( "Addr31","Addr31",
         0UL, 0x1UL,
         0, NULL,
         BX_PARAMRULE_NORULE,
         BX_CIBEH_ADDR31,
         yesno_table,
         BX_CIBEHGRP_6, Mephisto_CI_Beh2Hw));

    ADDPAR(( "Attr24","Attr24",
         0UL, 0x1UL,
         0, NULL,
         BX_PARAMRULE_NORULE,
         BX_CIBEH_ATTR24, yesno_table,
         BX_CIBEHGRP_7, Mephisto_CI_Beh2Hw));

    ADDPAR(( "Attr25","Attr25",
         0UL, 0x1UL,
         0, NULL,
         BX_PARAMRULE_NORULE,
         BX_CIBEH_ATTR25,
         yesno_table,
         BX_CIBEHGRP_7, Mephisto_CI_Beh2Hw));

    ADDPAR(( "Attr26","Attr26", 0UL, 0x1UL, 0, NULL, 0UL,
         BX_CIBEH_ATTR26, yesno_table,
         BX_CIBEHGRP_7, Mephisto_CI_Beh2Hw));

    ADDPAR(( "Attr27","Attr27", 0UL, 0x1UL, 0, NULL, 0UL,
         BX_CIBEH_ATTR27, yesno_table,
         BX_CIBEHGRP_7, Mephisto_CI_Beh2Hw));

    ADDPAR(( "Attr28","Attr28", 0UL, 0x1UL, 0, NULL, 0UL,
         BX_CIBEH_ATTR28, yesno_table,
         BX_CIBEHGRP_7, Mephisto_CI_Beh2Hw));

    ADDPAR(( "Errmessage", "ErrMsg",
         0x0UL, 0x1UL,
         0, NULL,
         BX_PARAMRULE_PPRVAR,
         BX_CIBEH_ERRMESSAGE,
         yesno_table,
         BX_CIBEHGRP_7, Mephisto_CI_Beh2Hw));

    ADDPAR(( "Partition","Partition",
         0x0UL, 63UL,
         BX_CIBEH_PARTITION_NO, NULL,
         BX_PARAMRULE_PPRVAR,
         BX_CIBEH_PARTITION,
         BX_CIBEH_PARTITION_table,
         BX_CIBEHGRP_3, Mephisto_CI_Beh2Hw));

    ADDPAR(( "Conditional Start", "CondStart",
         BX_CIBEH_CONDSTART_ONCE1, BX_CIBEH_CONDSTART_WAIT2,
         BX_CIBEH_CONDSTART_NO, NULL,
         BX_PARAMRULE_ZEROALLOWED,
         BX_CIBEH_CONDSTART,
         BX_CIBEH_CONDSTART_table,
         BX_CIBEHGRP_8, Mephisto_CI_Beh2Hw));

    ADDPAR(( "Delay","Delay", 
         0x1UL, 65535UL,
         1, NULL,
         BX_PARAMRULE_PPRVAR,
         BX_CIBEH_DELAY,
         NULL,
         BX_CIBEHGRP_2 | BX_CIBEHGRP_4 | BX_CIBEHGRP_5 , Mephisto_CI_Beh2Hw));

    ADDPAR(( "Steps","Steps",
         0x0UL, 0x4UL,  /* 6 for non compliant behavior ??? */
         2, NULL,
         BX_PARAMRULE_PPRVAR,
         BX_CIBEH_STEPS,
         NULL,
         BX_CIBEHGRP_6, Mephisto_CI_Beh2Hw));

    ADDPAR(( "Req64","Req64",
         0x0UL, 0x1UL,
         1, NULL,
         BX_PARAMRULE_PPRVAR,
         BX_CIBEH_REQ64,
         yesno_table,
         BX_CIBEHGRP_6, Mephisto_CI_Beh2Hw));

    ADDPAR(( "Release REQ#","RelReq",
         1UL, 2047UL,
         2047UL, NULL,
         BX_PARAMRULE_PPRVAR,
         BX_CIBEH_RELREQ,
         NULL,
         BX_CIBEHGRP_1 | BX_CIBEHGRP_2 , Mephisto_CI_Beh2Hw));


   /* REPEAT parameter has special off-by-one handling in xci.c */
    ADDPAR(( "Repeat","Repeat", 
         1UL, 256UL,
         1UL, NULL,
         BX_PARAMRULE_NORULE,
         BX_CIBEH_REPEAT, NULL,
         BX_CIBEHGRP_0, Mephisto_CI_Beh2Hw));

  }

  /* associate the right number of elements */
  pCIBehGen->num_elem = num_elem;

  pCapable->info[BX_PARAM_CI_BEH].paraminfo = p_anchor;  /* set it */


  /* -------------------------------------------------------------------- */
  /* the RT behavior proptery information (generic and parameters)      */
  /* -------------------------------------------------------------------- */

  /* short cut */
  pRTBehGen = pCapable->info[BX_PARAM_RT_BEH].geninfo;

  /* group name */
  pRTBehGen->group_name = "Requester-Target Behavior";

  pRTBehGen->depth = 256;

  /* the targetbeh parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;
  if (1) /* lic & BX_CAPABILITY_EXERCISER) */
  {
    ADDPAR(( "Decode Speed","DecSpeed",
         BX_RTBEH_DECSPEED_A, (BestXHasMephisto(handle)?BX_RTBEH_DECSPEED_B:BX_RTBEH_DECSPEED_C), /* Mephisto bug: DECSPEED_C not allowed !!! */
         BX_RTBEH_DECSPEED_B, NULL,
         BX_PARAMRULE_PPRVAR,
         BX_RTBEH_DECSPEED,
         BX_RTBEH_DECSPEED_table,
         BX_RTBEHGRP_4, Mephisto_RT_Beh2Hw));

    ADDPAR(( "Ack64","Ack64",
         0x0UL, 0x1UL,
         1UL, NULL,
         BX_PARAMRULE_PPRVAR,
         BX_RTBEH_ACK64,
         yesno_table,
         BX_RTBEHGRP_4, Mephisto_RT_Beh2Hw));

    ADDPAR(( "Initial","Initial", 
         0x0UL, 0x3UL,
         BX_RTBEH_INITIAL_ACCEPT, NULL,
         BX_PARAMRULE_PPRVAR,
         BX_RTBEH_INITIAL,
         BX_RTBEH_INITIAL_table,
         BX_RTBEHGRP_1, Mephisto_RT_Beh2Hw));

    ADDPAR(( "Latency","Latency",
         3UL, 34UL,
         3UL, NULL,
         BX_PARAMRULE_PPRVAR,
         BX_RTBEH_LATENCY,
         NULL /* no table */,
         BX_RTBEHGRP_1, Mephisto_RT_Beh2Hw));

    ADDPAR(( "Subseq","Subseq",
         0x0UL, (BestXHasMephisto(handle)?0x1:0x2),
         BX_RTBEH_SUBSEQ_ACCEPT, NULL,
         BX_PARAMRULE_PPRVAR,
         BX_RTBEH_SUBSEQ,
         BX_RTBEH_SUBSEQ_table,
         BX_RTBEHGRP_3, Mephisto_RT_Beh2Hw));

    ADDPAR(( "Subseqphase","SubseqPhase",
         0x0UL, 2047UL,
         0, NULL,
         BX_PARAMRULE_PPRVAR,
         BX_RTBEH_SUBSEQPHASE,
         NULL /* notable */,
         BX_RTBEHGRP_2 | BX_RTBEHGRP_3, Mephisto_RT_Beh2Hw));

    ADDPAR(( "Repeat","Repeat",
         0x1UL, 0x10000UL,
         1, NULL,
         BX_PARAMRULE_NORULE,
         BX_RTBEH_REPEAT,
         NULL,
         BX_RTBEHGRP_0, Mephisto_RT_Beh2Hw));
  }

  /* associate the right number of elements */
  pRTBehGen->num_elem = num_elem;

  pCapable->info[BX_PARAM_RT_BEH].paraminfo = p_anchor;  /* set it */


  /* -------------------------------------------------------------------- */
  /* the requester generic property information (generic and parameters) */
  /* -------------------------------------------------------------------- */

  /* short cut */
  pRTGen = pCapable->info[BX_PARAM_RT_GEN].geninfo;

  /* leave depth of the target gen memory = 0 (there is no such memory) */
  pRTGen->group_name = "Requester-Target Generic";

  /* the targetgen parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (1) /* lic & BX_CAPABILITY_EXERCISER ||
            lic & BX_CAPABILITY_HOSTINT) */
  {
    ADDPAR(( "BX_RTGEN_NUMBEH", NULL, 1UL, 256UL,1UL,
         NULL, 0UL, BX_RTGEN_NUMBEH,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_RTGEN_NUMBEHG0", NULL, 1UL, 256UL,1UL,
         NULL, 0UL, BX_RTGEN_NUMBEHG0,
         NULL, BX_RTBEHGRP_0 , NULL));

    ADDPAR(( "BX_RTGEN_NUMBEHG1", NULL, 1UL, 256UL,1UL,
         NULL, 0UL, BX_RTGEN_NUMBEHG1,
         NULL, BX_RTBEHGRP_1 , NULL));

    ADDPAR(( "BX_RTGEN_NUMBEHG2", NULL, 1UL, 256UL,1UL,
         NULL, 0UL, BX_RTGEN_NUMBEHG2,
         NULL, BX_RTBEHGRP_2 , NULL));

    ADDPAR(( "BX_RTGEN_NUMBEHG3", NULL, 1UL, 256UL,1UL,
         NULL, 0UL, BX_RTGEN_NUMBEHG3,
         NULL, BX_RTBEHGRP_3 , NULL));

    ADDPAR(( "BX_RTGEN_NUMBEHG4", NULL, 1UL, 256UL,1UL,
         NULL, 0UL, BX_RTGEN_NUMBEHG4,
         NULL, BX_RTBEHGRP_4 , NULL));
  }

  /* associate the right number of elements */
  pRTGen->num_elem = num_elem;

  pCapable->info[BX_PARAM_RT_GEN].paraminfo = p_anchor; /* set it */


  /*--------------------------------------------------------------------- */
  /*--------------------------------------------------------------------- */
  /*--------------------------------------------------------------------- */
  /*--------------------Start of Decoder--------------------------------- */
  /*--------------------------------------------------------------------- */
  /*--------------------------------------------------------------------- */
  /*--------------------------------------------------------------------- */

  /* short cut */
  pDecoderGen = pCapable->info[BX_PARAM_DECODER_GEN].geninfo;

  pDecoderGen->group_name = "Decoder";

  /* the decoder parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (1)
  {
    ADDPAR(( "Bar 0", "bar0",
         0x0UL, 0x0UL, 0UL, NULL,
         BX_PARAMRULE_NORULE, BX_DEC_BAR0,
         NULL,
         BX_PARAM_DECODER_STD, NULL));

    ADDPAR(( "Bar 1", "bar1",
         0x0UL, 0x0UL, 0UL, NULL,
         BX_PARAMRULE_NORULE, BX_DEC_BAR1,
         NULL,
         BX_PARAM_DECODER_STD, NULL));

    ADDPAR(( "Bar 2", "bar2",
         0x0UL, 0x0UL, 0UL, NULL,
         BX_PARAMRULE_NORULE, BX_DEC_BAR2,
         NULL,
         BX_PARAM_DECODER_STD, NULL));

    ADDPAR(( "Bar 3", "bar3",
         0x0UL, 0x0UL, 0UL, NULL,
         BX_PARAMRULE_NORULE, BX_DEC_BAR3,
         NULL,
         BX_PARAM_DECODER_STD, NULL));

    ADDPAR(( "Bar 4", "bar4",
         0x0UL, 0x0UL, 0UL, NULL,
         BX_PARAMRULE_NORULE, BX_DEC_BAR4,
         NULL,
         BX_PARAM_DECODER_STD, NULL));

    ADDPAR(( "Bar 5", "bar5",
         0x0UL, 0x0UL, 0UL, NULL,
         BX_PARAMRULE_NORULE, BX_DEC_BAR5,
         NULL,
         BX_PARAM_DECODER_STD, NULL));


    ADDPAR(( "Config", "config",
         0x0UL, 0x0UL, 0UL, NULL,
         BX_PARAMRULE_NORULE, BX_DEC_CONFIG,
         NULL,
         BX_PARAM_DECODER_CFG, NULL));

    ADDPAR(( "Expansion ROM", "exprom",
         0x0UL, 0x0UL, 0UL, NULL,
         BX_PARAMRULE_NORULE, BX_DEC_EXPROM,
         NULL,
         BX_PARAM_DECODER_EXP, NULL));


    ADDPAR(( "Requester Target", "rtdec",
         0x0UL, 0x0UL, 0UL, NULL,
         BX_PARAMRULE_NORULE, BX_DEC_RT,
         NULL,
         BX_PARAM_DECODER_REQ, NULL));
  }

  /* associate the right number of elements */
  pDecoderGen->num_elem = num_elem;

  pCapable->info[BX_PARAM_DECODER_GEN].paraminfo = p_anchor; /* set it */


  /*----------------------------------------------------------------------*
   *------------------- info about standard decoders ---------------------*
   *----------------------------------------------------------------------*/

  /* short cut */
  pDecoderStdGen = pCapable->info[BX_PARAM_DECODER_STD].geninfo;

  pDecoderStdGen->group_name = "Standard Decoder";

  pDecoderStdGen->depth = 6; /* Bar0-5 */

  /* the decoder parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (1)
  {
    ADDPAR(( "Location", "location",
         0x0UL, 0x1UL, BX_DECP_LOCATION_MEM, NULL,
         BX_PARAMRULE_MAXLONGALLOWED, BX_DECP_LOCATION,
         BX_DECP_LOCATION_table,
         0UL, NULL));

    /* Caution: Location should be added first.
       Else, setting e.g. the size from
       BestXTDecoderDefaultSet()
       may return an ERROR, because the
       wanted size cannot be set for the
       current decoder.
    */

    ADDPAR(( "Size", "size",
         2UL, 63UL, 12UL, NULL,
         BX_PARAMRULE_ZEROALLOWED,
         BX_DECP_SIZE,
         NULL,
         0UL, NULL));

    ADDPAR(( "Prefetch", "prefetch",
         0x0UL, 0x1UL, 0UL, NULL,
         BX_PARAMRULE_NORULE,
         BX_DECP_PREFETCH,
         onoff_table,
         0UL, NULL));

    ADDPAR(( "Resource Base", "resbase",
         0x0UL,1<<24, 0x0UL, NULL,
         BX_PARAMRULE_DWORDBOUND,
         BX_DECP_RESBASE,
         NULL,
         0UL, NULL));

    ADDPAR(( "Resource Size", "ressize",
         2UL, 24UL, (BestXHasMephisto(handle)?20UL:22UL) /* datamemsize */, NULL,
         BX_PARAMRULE_NORULE,
         BX_DECP_RESSIZE,
         NULL,
         0UL, NULL));

    ADDPAR(( "Resource", "resource",
         0UL,
         ( 1 << BX_DECP_RESOURCE_MEM |
           1 << BX_DECP_RESOURCE_GEN |
           1 << BX_DECP_RESOURCE_CFG),
         BX_DECP_RESOURCE_MEM,
         NULL,
         BX_PARAMRULE_BITSET,
         BX_DECP_RESOURCE,
         BX_DECP_RESOURCE_table,
         0UL, NULL));
    
    ADDPAR(( "Compare", "compare",
         0UL, 1UL, 0UL, NULL,
         BX_PARAMRULE_NORULE,
         BX_DECP_COMPARE,
         onoff_table,
         0UL, NULL));
  
    ADDPAR(( "Base Address", "baseaddr",
         0UL, 0xFFFFFFFF, 0x0UL,
         NULL,
         BX_PARAMRULE_NORULE,
         BX_DECP_BASEADDR,
         NULL /* no table */,
         0UL, NULL));

    ADDPAR(( "Base Address Hi", "baseaddrhi",
         0UL, 0xFFFFFFFF, 0x0UL,
         NULL,
         BX_PARAMRULE_NORULE,
         BX_DECP_BASEADDR_HI,
         NULL /* no table */,
         0UL, NULL));
  }

  /* associate the right number of elements */
  pDecoderStdGen->num_elem = num_elem;

  pCapable->info[BX_PARAM_DECODER_STD].paraminfo = p_anchor; /* set it */

  /* -------------------------------------------------------------------- */
  /* the decoder information for expansion rom decoder */
  /* -------------------------------------------------------------------- */

  /* short cut */
  pDecoderExpGen = pCapable->info[BX_PARAM_DECODER_EXP].geninfo;

  /* the number of expansion rom decoders  */
  pDecoderExpGen->depth = 1;    /* only one */
  pDecoderExpGen->group_name = "Expansion Rom Decoder";

  /* the decoder parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (1) /* lic & BX_CAPABILITY_EXERCISER) */
  {
    /* PCI spec:
       Maximum size is 24 (16MB)
       Minimum size is 11 (2KByte)
    */
    ADDPAR(( "Size", "size",
         11UL, 24UL, 0UL, NULL,BX_PARAMRULE_ZEROALLOWED,
         BX_DECP_SIZE,
         NULL,
         0UL, NULL));

    ADDPAR(( "Resource Base", "resbase",
         0x0UL, 1UL<<24, 0x0UL, NULL,
         BX_PARAMRULE_DWORDBOUND,
         BX_DECP_RESBASE,
         NULL,
         0UL, NULL));

    ADDPAR(( "Resource Size", "ressize",
         2UL, 24UL, 20UL /* datamemsize */, NULL,
         BX_PARAMRULE_NORULE,
         BX_DECP_RESSIZE,
         NULL,
         0UL, NULL));

    ADDPAR(( "Resource", "resource",
               0UL,
         ( 1 << BX_DECP_RESOURCE_MEM |
           1 << BX_DECP_RESOURCE_GEN |
           1 << BX_DECP_RESOURCE_FLASH ),
         BX_DECP_RESOURCE_MEM,
         NULL,
         BX_PARAMRULE_BITSET,
         BX_DECP_RESOURCE,
         BX_DECP_RESOURCE_table,
         0UL, NULL));

    ADDPAR(( "Base Address", "baseaddr",
         0UL, 0xFFFFFFFF, 0x0UL,
         NULL,
         BX_PARAMRULE_NORULE,
         BX_DECP_BASEADDR,
         NULL /* no table */,
         0UL, NULL));
  }

  /* associate the right number of elements */
  pDecoderExpGen->num_elem = num_elem;

  pCapable->info[BX_PARAM_DECODER_EXP].paraminfo = p_anchor; /* set it */


  /* -------------------------------------------------------------------- */
  /* the decoder information for config decoder */
  /* -------------------------------------------------------------------- */

  /* short cut */
  pDecoderCfgGen = pCapable->info[BX_PARAM_DECODER_CFG].geninfo;

  pDecoderCfgGen->group_name = "Config Decoder";

  pDecoderCfgGen->depth = 1;

  /* the decoder parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (1) /*lic & BX_CAPABILITY_EXERCISER) */
  {
    ADDPAR(( "Size", "size",
         0x0UL, 1, 1UL, NULL,BX_PARAMRULE_NORULE,
         BX_DECP_SIZE,
         NULL,
         0UL, NULL));
  }

  /* associate the right number of elements */
  pDecoderCfgGen->num_elem = num_elem;

  pCapable->info[BX_PARAM_DECODER_CFG].paraminfo = p_anchor; /* set it */

  /* -------------------------------------------------------------------- */
  /* the decoder information for requester decoder */
  /* -------------------------------------------------------------------- */

  /* short cut */
  pDecoderReqGen = pCapable->info[BX_PARAM_DECODER_REQ].geninfo;

  pDecoderReqGen->group_name = "Requester Decoder";

  pDecoderReqGen->depth = 1;

  /* the decoder parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (1) /* lic & BX_CAPABILITY_EXERCISER) */
  {
    ADDPAR(( "Size", "size",
         0x0UL, 1, 1UL, NULL,BX_PARAMRULE_NORULE,
         BX_DECP_SIZE,
         NULL,
         0UL, NULL));
  }

  /* associate the right number of elements */
  pDecoderReqGen->num_elem = num_elem;

  pCapable->info[BX_PARAM_DECODER_REQ].paraminfo = p_anchor; /* set it */

  /*--------------------------------------------------------------------- */
  /*--------------------------------------------------------------------- */
  /*--------------------------------------------------------------------- */
  /*--------------------End of Decoder----------------------------------- */
  /*--------------------------------------------------------------------- */
  /*--------------------------------------------------------------------- */
  /*--------------------------------------------------------------------- */

  /* -------------------------------------------------------------------- */
  /* the datamemory information (generic only) */
  /* -------------------------------------------------------------------- */

  /* short cut */
  pDatamemoryGen = pCapable->info[BX_PARAM_DATA_MEM].geninfo;

  pDatamemoryGen->group_name = "Data Memory";

  /* int_mem_offset is always 65536L */
  pDatamemoryGen->int_mem_offset = 65536L;
  switch(theBoard)
  {
    case BX_HW_E2929A:
    case BX_HW_E2929A_DEEP:
    case BX_HW_E2929B:
    case BX_HW_E2929B_DEEP:
    case BX_HW_E2922A:
    case BX_HW_E2922B:
      pDatamemoryGen->depth = 0x100000L; /* 2 * 128K * 32 bit = 1 MB */
      break;
    case BX_HW_E2930A:
    case BX_HW_E2930B:
    case BX_HW_E2923A:
    case BX_HW_E2923B:
    case BX_HW_E2923F:
      pDatamemoryGen->depth = 0x400000L; /* 2 * 512K * 32 bit = 4 MB */
      break;
    default: 
        BX_ERRETURN(BX_E_INVALID_CASE);
  } /*switch*/

  pDatamemoryGen->num_blocks = 0x7; /* qword alignment with busaddress for
                     * transfers */

  /* -------------------------------------------------------------------- */
  /* the trace generic property information (generic and parameters) */
  /* moved after the tracemem stuff so we know the tracememsize */
  /* -------------------------------------------------------------------- */


  /* -------------------------------------------------------------------- */
  /* the tracememoryprop information (generic and signals) */
  /* the number of bytes per trace memory line is stored in num_pages */
  /* -------------------------------------------------------------------- */

  /* short cut */
  pTracememGen = pCapable->info[BX_PARAM_TRACE_MEM].geninfo;

  pTracememGen->group_name = "Trace Memory Signals";

  /* NOTE; the number of bytes per line is in the num_pages field */
  switch(theBoard)
  {
    case BX_HW_E2929A:
    case BX_HW_E2929A_DEEP:
    case BX_HW_E2929B:
    case BX_HW_E2929B_DEEP:
      pTracememGen->num_pages = 16; /* 128 bit per line */
      tracememdepth = 0x200000;     /* 2Meg samples     */
      pCapable->info[BX_PARAM_TRACE_MEM].paraminfo = e2929a_signal_table;
      break;
    case BX_HW_E2922A:
    case BX_HW_E2922B:
      pTracememGen->num_pages = 16; /* 128 bit per line */
      tracememdepth = 0x0;          
      pCapable->info[BX_PARAM_TRACE_MEM].paraminfo = e2929a_signal_table;
      break;

    case BX_HW_E2930A:
    case BX_HW_E2930B:
      // Mode 1
      pTracememGen->num_pages = 16; /* 128 bit (used) per line */
      tracememdepth = 0x400000;     /* 4Meg samples            */
      pCapable->info[BX_PARAM_TRACE_MEM].paraminfo = e2929a_signal_table;

      if (BestXIsMode2(handle))
      {
        // Mode 2 
        pTracememGen->num_pages = 24; /* 192 bit per line MODE 2 DDR */
        pCapable->info[BX_PARAM_TRACE_MEM].paraminfo = e2930a_signal_table;
      
        if (BestXIsQDR(handle))
        {
          /* QDR: 
             Problem: PCI cycle does not correspond to one traceline any more.
             Need 320 bit per PCI cycle, i.e. two lines of trace memory
             per PCI cycle
          */
          /* pTracememGen->num_pages = 40;  320 bit per line  ???  */
          /* tracememdepth = 0x200000;      2Meg samples      ???  */
        }    
      }

      break;

    case BX_HW_E2923A:
    case BX_HW_E2923B:
    case BX_HW_E2923F:
      pTracememGen->num_pages = 24; /* 192 bit per line */
      tracememdepth = 0x0;     
      pCapable->info[BX_PARAM_TRACE_MEM].paraminfo = e2930a_signal_table;
      break;

    default: 
        BX_ERRETURN(BX_E_INVALID_CASE);
  } /*switch*/

  pTracememGen->depth = tracememdepth;

  /* count the number of elements within the list of signals */
  /* every list has a terminating line (important !! ) */
  num_elem = 0;
  if (pCapable->info[BX_PARAM_TRACE_MEM].paraminfo)
  {
    while (pCapable->info[BX_PARAM_TRACE_MEM].paraminfo[(int) num_elem++].param_name);  
  }

  pTracememGen->num_elem = num_elem - 1;


  /* -------------------------------------------------------------------- */
  /* MUST BE AFTER TRACEMEM stuff!!!! VERY important to access tracememdepth */
  /* the trace generic property information (generic and parameters) */
  /* -------------------------------------------------------------------- */

  /* short cut */
  pTraceGenGen = pCapable->info[BX_PARAM_TRACE_GEN].geninfo;

  pTraceGenGen->group_name = "Trace Memory";

  /* the triggerseq generic parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (1)  /* lic & BX_CAPABILITY_ANALYZER) */
  {
    ADDPAR(( "Heartbeat Mode", "heartbeatmode", 0x0UL, 0x1UL,BX_TRACE_HEARTBEATMODE_OFF, NULL, 0UL,
         BX_TRACE_HEARTBEATMODE,
         NULL, 0UL, NULL));

    ADDPAR(( "Heartbeat Value", "heartbeatvalue", 0x0UL, 0xffffffffUL,0x80000, NULL, 0UL,
         BX_TRACE_HEARTBEATVALUE,
         NULL, 0UL, NULL));


    /* SW is zero based, i.e property is number of lines stored _after_ the 
       triggerpoint. HW needs at least a '1' (denotes triggerpoint).  */
    ADDPAR(( "Trigger History", "trighistory", 0x0UL, tracememdepth-1,
         tracememdepth/2, NULL,
         (bx_int32) 0L,
         BX_TRACE_TRIG_HISTORY,
         NULL, 0UL, NULL));
  }

  /* associate the right number of elements */
  pTraceGenGen->num_elem = num_elem;

  pCapable->info[BX_PARAM_TRACE_GEN].paraminfo = p_anchor;  /* set it */


  /* -------------------------------------------------------------------- */
  /* the patternproperty information (generic and signals) */
  /* -------------------------------------------------------------------- */

  /* short cut */
  pPatternGen = pCapable->info[BX_PARAM_PATTERN].geninfo;

  pPatternGen->group_name = "Pattern Generic";

  pPatternGen->depth = 24;

  /* in this case we use the public table (pattdefs.h) to create a pattern
   * table for dynamic capabilities */


  p_anchor = NULL;            /* set to NULL to make AddParam Fct start
                               * with malloc */
  num_elem = 0;
  zwcnt = 0;

  /* patterns are used by analyzer and exerciser !!! */
  if (lic & (BX_CAPABILITY_ANALYZER|BX_CAPABILITY_EXERCISER))
  {
    bx_patt_infotype *patt_signal_table=NULL;

    switch(theBoard)
    {
      case BX_HW_E2929A:
      case BX_HW_E2929A_DEEP:
      case BX_HW_E2929B:
      case BX_HW_E2929B_DEEP:
      case BX_HW_E2922A:
      case BX_HW_E2922B:
        patt_signal_table=e2929a_patt_signal_table;
        break;
      case BX_HW_E2930A:
      case BX_HW_E2930B:
      case BX_HW_E2923A:
      case BX_HW_E2923B:
      case BX_HW_E2923F:
#if 1
        /* BESTX_PRINTF("Set MODE\n"); */

        if (BestXIsMode2(handle))
        {
          patt_signal_table=e2930a_patt_signal_table;
        }
        else
        {
          patt_signal_table=e2929a_patt_signal_table;
        }
        /* BESTX_PRINTF("Set MODE OK\n"); */

#else
        /* no mode dependence here ! */
        patt_signal_table=e2930a_patt_signal_table;
#endif
        break;
      default: 
        BX_ERRETURN(BX_E_INVALID_CASE);
    } /*switch*/

    while (patt_signal_table[zwcnt].Name)
    {
      if (patt_signal_table[zwcnt].GUIUsageCode) /* only if used at all */
      {

        GUIUsageCode=patt_signal_table[zwcnt].GUIUsageCode;


        ADDPAR(( patt_signal_table[zwcnt].Name,
         NULL,
         patt_signal_table[zwcnt].MinVal,
         patt_signal_table[zwcnt].MaxVal,
         patt_signal_table[zwcnt].BitCount,
         NULL,
         GUIUsageCode,
         0x0UL,
         patt_signal_table[zwcnt].StringList,
         0UL, NULL));
      }
      zwcnt++;
    }
  }

  /* associate the right number of elements */
  pPatternGen->num_elem = num_elem;

  pCapable->info[BX_PARAM_PATTERN].paraminfo = p_anchor; /* set it */

  /* count the number of elements within the list of signals */
  /* every list has a terminating line (important !! ) */

  /* -------------------------------------------------------------------- */
  /* the trigger sequencer generic information (generic and parameters) */
  /* -------------------------------------------------------------------- */

  /* short cut */
  pSeqTriggerGenGen = pCapable->info[BX_PARAM_TRIGSEQ_GEN].geninfo;

  pSeqTriggerGenGen->group_name = "Trigger Sequencer Generic";

  /* the triggerseq generic parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (1) /*lic & BX_CAPABILITY_ANALYZER) */
  {
    ADDPAR(( "BX_TRIGGEN_FBA", "FBA", 0x0UL, 0xffffffffUL, 0xffffffffUL, NULL, 0UL,
         BX_TRIGGEN_FBA,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_TRIGGEN_FBB", "FBB", 0x0UL, 0xffffffffUL, 0xffffffffUL, NULL, 0UL,
         BX_TRIGGEN_FBB,
         NULL, 0UL, NULL));
  }

  /* associate the right number of elements */
  pSeqTriggerGenGen->num_elem = num_elem;

  pCapable->info[BX_PARAM_TRIGSEQ_GEN].paraminfo = p_anchor; /* set it */

  /* -------------------------------------------------------------------- */
  /* the trigger sequencer tran information (generic and parameters) */
  /* -------------------------------------------------------------------- */

  /* short cut */
  pSeqTriggerTranGen = pCapable->info[BX_PARAM_TRIGSEQ_TRAN].geninfo;

  pSeqTriggerTranGen->group_name = "Trigger Sequencer Tran";

  pSeqTriggerTranGen->depth = 128;

  /* the triggerseq parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (1) /* lic & BX_CAPABILITY_ANALYZER) */
  {
    ADDPAR(( "BX_TRIGTRAN_STATE", "state", 0x0UL, 15, 0UL, NULL, 0UL,
         BX_TRIGTRAN_STATE,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_TRIGTRAN_NEXTSTATE", "nextstate", 0x0UL, 15, 0x0UL, NULL, 0UL,
         BX_TRIGTRAN_NEXTSTATE,
         NULL, 0UL, NULL));
  }

  /* associate the right number of elements */
  pSeqTriggerTranGen->num_elem = num_elem;

  pCapable->info[BX_PARAM_TRIGSEQ_TRAN].paraminfo = p_anchor; /* set it */


  /* -------------------------------------------------------------------- */
  /* the trigger sequencer cond information (generic and parameters) */
  /* -------------------------------------------------------------------- */

  /* short cut */
  pSeqTriggerCondGen = pCapable->info[BX_PARAM_TRIGSEQ_COND].geninfo;

  pSeqTriggerCondGen->group_name = "Trigger Sequencer Cond";

  pSeqTriggerCondGen->depth = 128;

  /* the triggerseq parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (1) /* lic & BX_CAPABILITY_ANALYZER) */
  {
    ADDPAR(( "BX_TRIGCOND_X", "x", 0x0UL, 0x0UL, 0x0UL, "0", 0x0UL,
         BX_TRIGCOND_X,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_TRIGCOND_TRIG", "trig", 0x0UL, 0x0UL, 0UL, "0", 0UL,
         BX_TRIGCOND_TRIG,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_TRIGCOND_SQ", "sq", 0x0UL, 0x0UL, 0x0UL, "0", 0x0UL,
         BX_TRIGCOND_SQ,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_TRIGCOND_FBADEC", "fbadec", 0x0UL, 0x0UL, 0UL, "0", 0UL,
         BX_TRIGCOND_FBADEC,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_TRIGCOND_FBBDEC", "fbbdec", 0x0UL, 0x0UL, 0UL, "0", 0UL,
         BX_TRIGCOND_FBBDEC,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_TRIGCOND_FBALOAD", "fbaload", 0x0UL, 0x0UL, 0UL, "0", 0UL,
         BX_TRIGCOND_FBALOAD,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_TRIGCOND_FBBLOAD", "fbbload", 0x0UL, 0x0UL, 0UL, "0", 0UL,
         BX_TRIGCOND_FBBLOAD,
         NULL, 0UL, NULL));
  }

  /* associate the right number of elements */
  pSeqTriggerCondGen->num_elem = num_elem;

  pCapable->info[BX_PARAM_TRIGSEQ_COND].paraminfo = p_anchor; /* set it */


  /* -------------------------------------------------------------------- */
  /* the performance sequencer generic information (generic and parameters) */
  /* -------------------------------------------------------------------- */

  /* short cut */
  pSeqPerforGenGen = pCapable->info[BX_PARAM_PERFSEQ_GEN].geninfo;

  pSeqPerforGenGen->group_name = "Performance Sequencer Generic";

  /* the triggerseq generic parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (1) /*lic & BX_CAPABILITY_ANALYZER) */
  {
    ADDPAR(( "BX_PERFGEN_FBA", "FBA", 0x0UL, 0xffffffffUL, 0xffffffffUL, NULL, 0UL,
         BX_PERFGEN_FBA,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_PERFGEN_CTRAMODE", "ctramode", 0x0UL, 0x1UL, BX_PERFGEN_CTRAMODE_INC, NULL, 0UL,
         BX_PERFGEN_CTRAMODE,
         NULL, 0UL, NULL));
  }

  /* associate the right number of elements */
  pSeqPerforGenGen->num_elem = num_elem;

  pCapable->info[BX_PARAM_PERFSEQ_GEN].paraminfo = p_anchor;  /* set it */


  /* -------------------------------------------------------------------- */
  /* the performance sequencer tran information (generic and parameters) */
  /* -------------------------------------------------------------------- */

  /* short cut */
  pSeqPerforTranGen = pCapable->info[BX_PARAM_PERFSEQ_TRAN].geninfo;

  pSeqPerforTranGen->group_name = "Performance Sequencer Tran";

  pSeqPerforTranGen->depth = 128;

  /* the triggerseq parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (1) /* lic & BX_CAPABILITY_ANALYZER) */
  {
    ADDPAR(( "BX_PERFTRAN_STATE", "state", 0x0UL, 15, 0UL, NULL, 0UL,
         BX_PERFTRAN_STATE,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_PERFTRAN_NEXTSTATE", "nextstate", 0x0UL, 15, 0x0UL, NULL, 0UL,
         BX_PERFTRAN_NEXTSTATE,
         NULL, 0UL, NULL));
  }

  /* associate the right number of elements */
  pSeqPerforTranGen->num_elem = num_elem;

  pCapable->info[BX_PARAM_PERFSEQ_TRAN].paraminfo = p_anchor;  /* set it */

  /* -------------------------------------------------------------------- */
  /* the performance sequencer cond information (generic and parameters) */
  /* -------------------------------------------------------------------- */

  /* short cut */
  pSeqPerforCondGen = pCapable->info[BX_PARAM_PERFSEQ_COND].geninfo;

  pSeqPerforCondGen->group_name = "Performance Sequencer Cond";

  pSeqPerforCondGen->depth = 128;

  /* the triggerseq parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (1) /* lic & BX_CAPABILITY_ANALYZER) */
  {
    ADDPAR(( "BX_PERFCOND_X", "x", 0x0UL, 0x0UL, 0x0UL, "0", 0x0UL,
         BX_PERFCOND_X,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_PERFCOND_CTRAINC", "ctrainc", 0x0UL, 0x0UL, 0UL, "0", 0UL,
         BX_PERFCOND_CTRAINC,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_PERFCOND_CTRBINC", "ctrbinc", 0x0UL, 0x0UL, 0UL, "0", 0UL,
         BX_PERFCOND_CTRBINC,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_PERFCOND_FBADEC", "fbadec", 0x0UL, 0x0UL, 0UL, "0", 0UL,
         BX_PERFCOND_FBADEC,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_PERFCOND_FBALOAD", "fbaload", 0x0UL, 0x0UL, 0UL, "0", 0UL,
         BX_PERFCOND_FBALOAD,
         NULL, 0UL, NULL));
  }

  /* associate the right number of elements */
  pSeqPerforCondGen->num_elem = num_elem;

  pCapable->info[BX_PARAM_PERFSEQ_COND].paraminfo = p_anchor;  /* set it */


  /* -------------------------------------------------------------------- */
  /* -------------------------------------------------------------------- */
  /* the observer rules information (generic and the rules itself) */
  /* the paramrule information holds the info for the bit position ! */
  /* -------------------------------------------------------------------- */
  /* -------------------------------------------------------------------- */

  /* short cut */
  pObserverRulesGen = pCapable->info[BX_PARAM_OBSRULES].geninfo;

  /* leave depth of the observer rules at 0 (what else would make sense?) */
  pObserverRulesGen->group_name = "Observer Rules";

  /* the observerrules rules information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  ADDPAR(( " 0: INITIATOR_0",
        "BX_OBSRULE_INITIATOR_0: An initiator can terminate a transaction with initiator abort (deassert FRAME# and IRDY#) 8 clocks after the address phase(s). PCI-X Spec 2.11.1.2",
        0UL, 1UL,
        BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_INITIATOR_0,
        NULL, 0UL, NULL));

  ADDPAR(( " 1: INITIATOR_1",
        "BX_OBSRULE_INITIATOR_1: If the target inserts wait states on burst write or split completion, the initiator must toggle between its first and second data values until the target asserts TRDY# (or terminates the transaction). PCI-X Spec 1.10.2 Rule 5",
        0UL, 1UL,
        BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_INITIATOR_1,
        NULL, 1UL, NULL));

  ADDPAR(( " 2: INITIATOR_2",
        "BX_OBSRULE_INITIATOR_2: The initiator is required to terminate the transaction when the byte count is satisfied. PCI-X Spec 1.10.2 Rule 6",
        0UL, 1UL,
        BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_INITIATOR_2,
        NULL, 2UL, NULL));

  ADDPAR(( " 3: INITIATOR_3",
        "BX_OBSRULE_INITIATOR_3: The initiator is permitted to disconnect a burst transaction (before the byte count is satisfied) only on an ADB. PCI-X Spec 1.10.2 Rule 7",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_INITIATOR_3,
         NULL, 3UL, NULL));

  ADDPAR(( " 4: TARGET_0",
        "BX_OBSRULE_TARGET_0: Burst Write and split completion transactions must not be terminated with split response. All other target terminations are permitted. PCI-X Spec 2.6.1",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_TARGET_0,
        NULL, 4UL, NULL));

  ADDPAR(( " 5: TARGET_1",
        "BX_OBSRULE_TARGET_1: The target is permitted to signal Single Data Phase Disconnect only on the first data phase (with or without preceding Wait States). PCI-X Spec 1.10.3 Rule 6",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_TARGET_1,
        NULL, 5UL, NULL));

  ADDPAR(( " 6: TARGET_2",
        "BX_OBSRULE_TARGET_2: The target is permitted to signal Split Response only on the first data phase (with or without preceding Wait States). PCI-X Spec 2.11.2.4",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_TARGET_2,
        NULL, 6UL, NULL));

  ADDPAR(( " 7: TARGET_3",
        "BX_OBSRULE_TARGET_3: Once the target has signaled Disconnect on Next ADB, it must continue to do so (or signal Target Abort) until the end of the transaction. PCI-X Spec 2.11.2.2",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_TARGET_3,
        NULL, 7UL, NULL));

  ADDPAR(( " 8: TARGET_4",
        "BX_OBSRULE_TARGET_4: The target deasserts DEVSEL#, STOP# and TRDY# one clock after the last data phase (if they are not already deasserted) and floats them one clock after that. PCI-X Spec 1.10.3 Rule 8",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_TARGET_4,
        NULL, 8UL, NULL));

  ADDPAR(( " 9: TARGET_5",
        "BX_OBSRULE_TARGET_5: There must be an even number of target initial wait states for a burst write and Split Completion. PCI-X Spec 2.9",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_TARGET_5,
        NULL, 9UL, NULL));

  ADDPAR(( "10: TARGET_6",
        "BX_OBSRULE_TARGET_6: If a PCI-X target signals Data Transfer (with or without preceding Wait States), the target is limited to disconnecting the transaction only on an ADB (until the byte count is satisfied). PCI-X Spec 1.10.3 Rule 5",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_TARGET_6,
        NULL, 10UL, NULL));

  ADDPAR(( "11: FRAME_0",
        "BX_OBSRULE_FRAME_0: FRAME# cannot be deasserted unless IRDY# was asserted. PCI Spec Appendix C, Rule 8c",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_FRAME_0,
        NULL, 11UL, NULL));

  ADDPAR(( "12: FRAME_1",
        "BX_OBSRULE_FRAME_1: When FRAME# has been deasserted, it cannot be reasserted during the same transaction. PCI spec Appendix C Rules 8b and 8d",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_FRAME_1,
        NULL, 12UL, NULL));

  ADDPAR(( "13: IRDY_0",
        "BX_OBSRULE_IRDY_0: IRDY# must be asserted two clocks after the attribute phase. PCI-X Spec 1.10.2. Rule 3b",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_IRDY_0,
         NULL, 13UL, NULL));

  ADDPAR(( "14: IRDY_1",
        "BX_OBSRULE_IRDY_1: Initiator Wait States are not permitted. PCI-X Spec 1.10.2. Rule 3b",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_IRDY_1,
        NULL, 14UL, NULL));

  ADDPAR(( "15: IRDY_2",
        "BX_OBSRULE_IRDY_2: A transaction starts when FRAME# is asserted for the first time. IRDY# must not go low when FRAME# is high. PCI Spec Appendix C, Rule 7 and 8c",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL, 
            BX_PARAMRULE_NORULE,
            BX_OBSRULE_IRDY_2,
        NULL, 15UL, NULL));

  ADDPAR(( "16: IRDY_3",
        "BX_OBSRULE_IRDY_3: Once asserted IRDY# must stay asserted until the end of transaction or till the target signals a termination. PCI-X Spec 1.10.2. Rule 3b and PCI-X Spec 2.9",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_IRDY_3,
        NULL, 16UL, NULL));

  ADDPAR(( "17: TRDY_0",
        "BX_OBSRULE_TRDY_0: TRDY# must not be asserted before the attribute phase, but two or more clocks later. PCI-X Spec 2.8 Table 2-6 and PCI Spec Appendix C Rule 14",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_TRDY_0,
        NULL, 17UL, NULL));

  ADDPAR(( "18: TRDY_1",
        "BX_OBSRULE_TRDY_1: Once TRDY# has been asserted, it must not be deasserted and reasserted during the same transaction (no subsequent wait states). PCI-X Spec 1.10.3 Rule 4",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_TRDY_1,
        NULL, 18UL, NULL));

  ADDPAR(( "19: DEVSEL_0",
        "BX_OBSRULE_DEVSEL_0: DEVSEL# must be asserted prior to the edge at which the target asserts TRDY#. PCI-X Spec 2.8 and PCI-X Spec 2.9.1",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_DEVSEL_0,
        NULL, 19UL, NULL));

  ADDPAR(( "20: DEVSEL_1",
        "BX_OBSRULE_DEVSEL_1: DEVSEL# must not be asserted during a special cycle or if a reserved command has been used. PCI-X Spec 2.4 and PCI-X Spec 2.7.3",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_DEVSEL_1,
        NULL, 20UL, NULL));

  ADDPAR(( "21: DEVSEL_2",
        "BX_OBSRULE_DEVSEL_2: DEVSEL# must not be asserted 1, 5 or more than 6 clocks after the address phase. PCI-X Spec 2.8",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_DEVSEL_2,
        NULL, 21UL, NULL));

  ADDPAR(( "22: DEVSEL_3",
        "BX_OBSRULE_DEVSEL_3: After a Target asserts DEVSEL#, it cannot be deasserted until the last data phase has completed, except to signal Data Transfer, Wait States, Target Abort, Split Response, Retry and Single Data Phase Disconnect. PCI-X Spec 1.10.3 Rule 3",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_DEVSEL_3,
        NULL, 22UL, NULL));

  ADDPAR(( "23: DEVSEL_4",
        "BX_OBSRULE_DEVSEL_4: DEVSEL# must be deasserted one clock after last transfer. PCI-X Spec 1.10.3 Rule 8",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_DEVSEL_4,
         NULL, 23UL, NULL));

  ADDPAR(( "24: STOP_0",
        "BX_OBSRULE_STOP_0: STOP# must not be asserted without DEVSEL# being asserted, except RST# being asserted. PCI-X Spec 1.10.1 Rule 12; PCI spec Appendix C, Rule 14, spec 6",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_STOP_0,
        NULL, 24UL, NULL));

  ADDPAR(( "25: LAT_0",
        "BX_OBSRULE_LAT_0: If the target signals Split Response, Target-Abort or Retry, the target must do so within eight clocks of the assertion of FRAME#.PCI-X Spec 1.10.3 Rule 4",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_LAT_0,
        NULL, 25UL, NULL));

  ADDPAR(( "26: LAT_1",
        "BX_OBSRULE_LAT_1: If the target signals Single Data Phase Disconnect, Data Transfer or Disconnect on Next ADB, the target must do so within 16 clocks of the assertion of FRAME#. PCI-X Spec 1.10.3 Rule 4",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_LAT_1,
        NULL, 26UL, NULL));

  ADDPAR(( "27: W64_0",
        "BX_OBSRULE_W64_0: ACK64# may only be asserted, when REQ64# was asserted before (ACK64# is a response to REQ64#). Mode 1 only. PCI Spec 3.8",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_W64_0,
        NULL, 27UL, NULL));

  ADDPAR(( "28: W64_1",
        "BX_OBSRULE_W64_1: A 64-bit initiator asserts REQ64# with the same timing as FRAME# to request a 64-bit data transfer. It deasserts REQ64# with FRAME# at the end of the transaction. Mode 1 only. PCI-X Spec 2.12.3 Requirement 4",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_W64_1,
        NULL, 28UL, NULL));

  ADDPAR(( "29: W64_2",
        "BX_OBSRULE_W64_2: If a 64-bit target is addressed by a transaction that does have REQ64# asserted with FRAME#, the target asserts ACK64# with DEVSEL# to complete the transaction as a 64-bit target. It deasserts ACK64# with DEVSEL# at the end of the transaction. Mode 1 only. PCI-X Spec 2.12.3",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_W64_2,
        NULL, 29UL, NULL));

  ADDPAR(( "30: W64_3",
        "BX_OBSRULE_W64_3: REQ64# must not be used with special cycle or interrupt acknowledge command. Only burst transactions (memory commands other than Memory read DWORD) use 64-bit data transfers. PCI-X Spec 2.4 and 2.7",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_W64_3,
        NULL, 30UL, NULL));

  ADDPAR(( "31: W64_4",
        "BX_OBSRULE_W64_4: For DWORD Transactions, REQ64# must be deasserted. PCI-X Spec 2.12.3 Requirement 2",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_W64_4,
        NULL, 31UL, NULL));

  ADDPAR(( "32: PAR_0",
        "BX_OBSRULE_PAR_0: PERR# may never be asserted three clocks after the address phase (or earlier in a transaction) or during a special cycle. During WRITE, PERR# may be asserted three clocks after IRDY#, during READ, PERR# may be asserted three clocks after TRDY#. Mode 1 only. PCI Spec 3.7.4",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_PAR_0,
        NULL, 32UL, NULL));

  ADDPAR(( "33: PAR_1",
        "BX_OBSRULE_PAR_1: AD[31::0] address parity error. Mode 1 only. PCI Spec Appendix C, Rule 32 b",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_PAR_1,
         NULL, 33UL, NULL));

  ADDPAR(( "34: PAR_2",
        "BX_OBSRULE_PAR_2: AD[63::32] address parity error. Mode 1 only. PCI Spec Appendix C, Rule 32 c",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_PAR_2,
        NULL, 34UL, NULL));

  ADDPAR(( "35: PAR_3",
        "BX_OBSRULE_PAR_3: AD[31::0] data parity error occurred but was not signaled. Mode 1 only. PCI-X Spec 5.3",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_PAR_3,
        NULL, 35UL, NULL));

  ADDPAR(( "36: PAR_4",
        "BX_OBSRULE_PAR_4: AD[63::32] data parity error occurred but was not signaled. Mode 1 only. PCI-X Spec 5.3",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_PAR_4,
        NULL, 36UL, NULL));

  ADDPAR(( "37: PAR_5",
        "BX_OBSRULE_PAR_5: AD[31::0] data parity error occurred. Mode 1 only. PCI Spec Appendix C, Rule 32 b",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_PAR_5,
        NULL, 37UL, NULL));

  ADDPAR(( "38: PAR_6",
        "BX_OBSRULE_PAR_6: AD[63::32] data parity error occurred. Mode 1 only. PCI Spec Appendix C, Rule 32 c",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_PAR_6,
        NULL, 38UL, NULL));

  ADDPAR(( "39: SEM_0",
        "BX_OBSRULE_SEM_0: For I/O and DWORD Memory transactions, AD[1::0] and byte enable encoding must have a defined relationship. PCI-X Spec 2.0, table 2-5",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_SEM_0,
        NULL, 39UL, NULL));

  ADDPAR(( "40: SEM_1",
        "BX_OBSRULE_SEM_1: Reserved commands are reserved for future use. PCI-X Spec 2",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_SEM_1,
        NULL, 40UL, NULL));

  ADDPAR(( "41: SEM_2",
        "BX_OBSRULE_SEM_2: DAC is not allowed immediately after a DAC. PCI Spec 3.9 and PCI-X Spec 2.12.1",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_SEM_2,
        NULL, 41UL, NULL));

  ADDPAR(( "42: SEM_3",
        "BX_OBSRULE_SEM_3: During the data phases C/BE# bus must be driven high for all Burst Transactions except Memory Write. PCI-X Spec 2.6",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_SEM_3,
        NULL, 42UL, NULL));

  ADDPAR(( "43: SEM_4",
        "BX_OBSRULE_SEM_4: During a Dual Address Cycle, a 64-bit initiator has to drive the upper half of the address on AD[63::32] in the initial and in the second address phase. Mode 1 only. PCI-X Spec 2.12.1.3 a i)",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_SEM_4,
         NULL, 43UL, NULL));

  ADDPAR(( "44: SEM_5",
        "BX_OBSRULE_SEM_5: In the second address phase of a Dual Address Cycle, AD [63:32] and AD [31:0] contain duplicate copies of the upper half of the address. Mode 1 only. PCI-X Spec 2.12.1.3 a i)",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_SEM_5,
        NULL, 44UL, NULL));

  ADDPAR(( "45: SEM_6",
        "BX_OBSRULE_SEM_6: During a Dual Address Cycle, a 64-bit initiator has to drive the bus command on C/BE [7::4]# in the initial and the second address phase. Mode 1 only. PCI-X Spec 2.12.1.3 a ii)",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_SEM_6,
        NULL, 45UL, NULL));

  ADDPAR(( "46: SEM_7",
        "BX_OBSRULE_SEM_7: In the second address phase of a Dual Address Cycle, C/BE [7::4]# and C/BE [3::0]# contain duplicate copies of the transaction command. Mode 1 only. PCI-X Spec 2.12.1.3 a ii)",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_SEM_7,
        NULL, 46UL, NULL));

  ADDPAR(( "47: SEM_8",
        "BX_OBSRULE_SEM_8: DWORD Transactions only support a single data phase. PCI-X Spec 2.7",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_SEM_8,
        NULL, 47UL, NULL));

  ADDPAR(( "48: SEM_9",
        "BX_OBSRULE_SEM_9: An initiator that supports 64-bit addressing must generate a SAC instead of a DAC, when the upper 32 bits of the address are zero. PCI spec 3.9",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_SEM_9,
        NULL, 48UL, NULL));

  ADDPAR(( "49: SEM_10",
        "BX_OBSRULE_SEM_10: This rule detects a bus hang. If the bus does not get idle once within 4095 clocks, it triggers. Agilent Rule to detect potential deadlocks.",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_SEM_10,
        NULL, 49UL, NULL));

  ADDPAR(( "50: SEM_11",
        "BX_OBSRULE_SEM_11: A Completer did not respond to a split transaction within 2^20-1 clocks. Agilent Rule to detect potential deadlocks.",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_SEM_11,
        NULL, 50UL, NULL));

  ADDPAR(( "51: SEM_12",
        "BX_OBSRULE_SEM_12: A delayed transaction (retries) has not been repeated within 2^15 clocks. PCI spec 3.3.3.3.3",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_SEM_12,
        NULL, 51UL, NULL));

  ADDPAR(( "52: SEM_13",
        "BX_OBSRULE_SEM_13: A delayed transaction has not terminated within 2^16 clocks. Agilent Rule to detect potential deadlocks.",
        0UL, 1UL, BX_OBS_MASK_MEPHISTO|BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_SEM_13,
        NULL, 52UL, NULL));
  ADDPAR(( "53: ECC_0",
        "BX_OBSRULE_ECC_0: ECC address error.",
        0UL, 1UL, BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_ECC_0,
        NULL, 53UL, NULL));
  ADDPAR(( "54: ECC_1",
        "BX_OBSRULE_ECC_1: ECC correctable error detected in dataphase.",
        0UL, 1UL, BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_ECC_1,
        NULL, 54UL, NULL));
  ADDPAR(( "55: ECC_2",
        "BX_OBSRULE_ECC_2: ECC uncorrectable error detected in dataphase.",
        0UL, 1UL, BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_ECC_2,
        NULL, 55UL, NULL));
  ADDPAR(( "56: ECC_3",
        "BX_OBSRULE_ECC_3: Uncorrectable ECC error detected in dataphase and not signaled.",
        0UL, 1UL, BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_ECC_3,
        NULL, 56UL, NULL));

#if 0 /* These not yet implemented in HW */
  ADDPAR(( "57: PAR_7",
        "BX_OBSRULE_PAR_7: PERR# was asserted when there was no parity or ECC error. PCI-X Spec 1.12.6 Rule 3",
        0UL, 1UL, BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_PAR_7,
        NULL, 57UL, NULL));
  ADDPAR(( "58: PAR_8",
        "BX_OBSRULE_PAR_8: Uncorrectable Address or Attribute Error detected. SERR# should get asserted on this clock, but is not. PCI-X Spec 5.2.3",
        0UL, 1UL, BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_PAR_8,
        NULL, 58UL, NULL));
  ADDPAR(( "59: SEM_14",
        "BX_OBSRULE_SEM_14: There are always at least 2 idle clocks between transactions in Mode 2 (including bus turn-around). Mode 2 only. PCI-X Spec 1.12.2 Rule 9",
        0UL, 1UL, BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_SEM_14,
        NULL, 59UL, NULL));
  ADDPAR(( "60: SEM_15",
        "BX_OBSRULE_SEM_15: In PCI-X Mode 2, a 32-bit target of a burst push transaction from a 64-bit initiator must insert a minimum of two wait states. Mode 2 only. PCI-X Spec 1.12.7 Rule 8",
        0UL, 1UL, BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_SEM_15,
        NULL, 60UL, NULL));
  ADDPAR(( "61: SEM_16",
        "BX_OBSRULE_SEM_16: This indicates the AD[6:0] is not zero for Split Completion of a DWORD Request or Split Completion Message. PCI-X Spec 2.10.2",
        0UL, 1UL, BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_SEM_16,
        NULL, 61UL, NULL));
  ADDPAR(( "62: SEM_17",
        "BX_OBSRULE_SEM_17: This indicates the byte count is not equal to four for Split Completion of a DWORD Request or for Split Completion Message. PCI-X Spec 2.10.2",
        0UL, 1UL, BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_SEM_17,
        NULL, 62UL, NULL));
  ADDPAR(( "63: SEM_18",
        "BX_OBSRULE_SEM_18: After the arbiter asserts GNT#, the arbiter must keep it asserted for a minimum of five clocks while the bus is idle, or until the initiator asserts FRAME# or deasserts REQ#. PCI-X Spec 1.12.4 Rule 4",
        0UL, 1UL, BX_OBS_MASK_MODE1|BX_OBS_MASK_MODE2, NULL,
        BX_PARAMRULE_NORULE,
        BX_OBSRULE_SEM_18,
        NULL, 63UL, NULL));
#endif


  /* associate the right number of elements */
  pObserverRulesGen->num_elem = num_elem;

  pCapable->info[BX_PARAM_OBSRULES].paraminfo = p_anchor; /* set it */

  /* -------------------------------------------------------------------- */
  /* -------------------------------------------------------------------- */
  /* the status information (generic and props )                          */
  /* -------------------------------------------------------------------- */
  /* -------------------------------------------------------------------- */

  /* short cut */
  pStatusGen = pCapable->info[BX_PARAM_STATUS].geninfo;

  pStatusGen->group_name = "Card Status";

  /* the status information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (1) /* lic & BX_CAPABILITY_ANALYZER) */
  {
    ADDPAR(( "BX_STAT_CMP_FAIL", "",
         0UL, 1UL, 0UL, NULL, 0UL, BX_STAT_CMP_FAIL,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_CMP_CMD", "",
         0UL, 15UL, 0UL, NULL, 0UL, BX_STAT_CMP_CMD,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_CMP_ADDR_HI", "",
         0UL, 0xffffffffUL, 0UL, NULL, 0UL, BX_STAT_CMP_ADDR_HI,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_CMP_ADDR_LO", "",
         0UL, 0xffffffffUL, 0UL, NULL, 0UL, BX_STAT_CMP_ADDR_LO ,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_CMP_EXTCMD", "",
         0UL, 15UL, 0UL, NULL, 0UL, BX_STAT_CMP_EXTCMD,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_CMP_ATTR_LO", "",
         0UL, 0xffffffffUL, 0UL, NULL, 0UL, BX_STAT_CMP_ATTR_LO,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_CMP_ACTUAL_HI", "",
         0UL, 0xffffffffUL, 0UL, NULL, 0UL, BX_STAT_CMP_ACTUAL_HI,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_CMP_ACTUAL_LO", "",
         0UL, 0xffffffffUL, 0UL, NULL, 0UL, BX_STAT_CMP_ACTUAL_LO,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_CMP_REF_HI", "",
         0UL, 0xffffffffUL, 0UL, NULL, 0UL, BX_STAT_CMP_REF_HI,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_CMP_REF_LO", "",
         0UL, 0xffffffffUL, 0UL, NULL, 0UL, BX_STAT_CMP_REF_LO,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_CMP_DATAPHASE", "",
         0UL, 0x7ffUL, 0UL, NULL, 0UL, BX_STAT_CMP_DATAPHASE,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_CMP_DATASUBPHASE", "",
         0UL, 0x3UL, 0UL, NULL, 0UL, BX_STAT_CMP_DATASUBPHASE,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_CMP_RESOURCE", "",
         0UL, 0x1UL, 0UL, NULL, 0UL, BX_STAT_CMP_RESOURCE,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_CMP_BEATTR", "",
         0UL, 0xfUL, 0UL, NULL, 0UL, BX_STAT_CMP_BEATTR,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_CMP_BEDATA", "",
         0UL, 0xffUL, 0UL, NULL, 0UL, BX_STAT_CMP_BEDATA,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_CMP_XFERSIZE", "",
         0UL, 0x1UL, 0UL, NULL, 0UL, BX_STAT_CMP_XFERSIZE,
         NULL, 0UL, NULL));
             
    ADDPAR(( "BX_STAT_CMP_ICMP", "",
         0UL, 0x1UL, 0UL, NULL, 0UL, BX_STAT_CMP_ICMP,
         NULL, 0UL, NULL));
             
    ADDPAR(( "BX_STAT_CMP_TCMP", "",
         0UL, 0x1UL, 0UL, NULL, 0UL, BX_STAT_CMP_TCMP,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_ERR_PERR", "",
         0UL, 1UL, 0UL, NULL, 0UL, BX_STAT_ERR_PERR ,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_ERR_SERR", "",
         0UL, 1UL, 0UL, NULL, 0UL, BX_STAT_ERR_SERR ,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_ERR_WRPAR", "",
         0UL, 1UL, 0UL, NULL, 0UL, BX_STAT_ERR_WRPAR,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_ERR_WRPAR64", "",
         0UL, 1UL, 0UL, NULL, 0UL, BX_STAT_ERR_WRPAR64 ,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_INTA", "",
         0UL, 1UL, 0UL, NULL, 0UL, BX_STAT_INTA,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_INTB", "",
         0UL, 1UL, 0UL, NULL, 0UL, BX_STAT_INTB ,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_INTC", "",
         0UL, 1UL, 0UL, NULL, 0UL, BX_STAT_INTC ,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_INTD", "",
         0UL, 1UL, 0UL, NULL, 0UL, BX_STAT_INTD,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_OBS_ERR", "",
         0UL, 1UL, 0UL, NULL, 0UL, BX_STAT_OBS_ERR,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_TRC_RUNNING", "",
         0UL, 1UL, 0UL, NULL, 0UL, BX_STAT_TRC_RUNNING ,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_TRC_TRIGGER", "",
         0UL, 1UL, 0UL, NULL, 0UL, BX_STAT_TRC_TRIGGER ,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_TRC_MEMFULL", "",
         0UL, 1UL, 0UL, NULL, 0UL, BX_STAT_TRC_MEMFULL,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_TRC_TRIGPOS", "",
         0UL, 0xffffffffUL, 0UL, NULL, 0UL, BX_STAT_TRC_TRIGPOS ,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_TRC_LINES", "",
         0UL, 0xffffffffUL, 0UL, NULL, 0UL, BX_STAT_TRC_LINES ,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_PCIXPOWER", "",
         0UL, 1UL, 1UL, NULL, 0UL, BX_STAT_PCIXPOWER ,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_PCIXMODE", "",
         0UL, 1UL, 1UL, NULL, 0UL, BX_STAT_PCIXMODE ,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_PCIRESET", "",
         0UL, 1UL, 0UL, NULL, 0UL, BX_STAT_PCIRESET ,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_RESETCODE", "",
         0UL, 7UL, 0UL, NULL, 0UL, BX_STAT_RESETCODE ,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_LASTRESET", "",
         0UL, 0xffffffffUL, 0UL, NULL, 0UL, BX_STAT_LASTRESET ,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_BUSSPEED", "",
         0UL, 0xffffffffUL, 0UL, NULL, 0UL, BX_STAT_BUSSPEED ,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_PLLSPEED", "",
         0UL, 0xffffffffUL, 0UL, NULL, 0UL, BX_STAT_PLLSPEED ,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_BUSWIDTH", "",
         0UL, 0xffffffffUL, 0UL, NULL, 0UL, BX_STAT_BUSWIDTH ,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_PIGGYBACKID", "",
         0UL, 3UL, 0UL, NULL, 0UL, BX_STAT_PIGGYBACKID,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_TRIGIO", "",
         0UL, 15UL, 0UL, NULL, 0UL, BX_STAT_TRIGIO ,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_INVISIBLE", "",
         0UL, 1UL, 0UL, NULL, 0UL, BX_STAT_INVISIBLE ,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_USRSETT", "",
         0UL, 1UL, 0UL, NULL, 0UL, BX_STAT_USRSETT ,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_STDRECOVR", "",
         0UL, 1UL, 0UL, NULL, 0UL, BX_STAT_STDRECOVR ,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_IABORT", "",
         0UL, 1UL, 0UL, NULL, 0UL, BX_STAT_IABORT ,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_TABORT", "",
         0UL, 1UL, 0UL, NULL, 0UL, BX_STAT_TABORT ,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_TEST", "",
         0UL, 0xffffffffUL, 0, NULL, 0UL, BX_STAT_TEST ,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_CFGSTAT", "",
         0UL, 0xffffffffUL, 0, NULL, 0UL, BX_STAT_CFGSTAT ,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_CLK", "",
         0UL, 1UL, 0UL, NULL, 0UL, BX_STAT_CLK ,
         NULL, 0UL, NULL));

    ADDPAR(( "BX_STAT_SPLITFAIL", "",
         0UL, 1UL, 0UL, NULL, 0UL, BX_STAT_SPLITFAIL ,
         NULL, 0UL, NULL));

    if(BestXHasFirmware(handle))
    {
      /* Still open:
         - Move HW implementation from Fred to Faust (new ASIC cut)
         - Connect I2C bus to Faust (new board layout)
         - Copy intialization code from FW to CAPI (for 23ers)
      */
      if (BestXHasFaustFPGA(handle))
      {
        /* 1618 only on E2930A */
        ADDPAR(( "BX_STAT_DIAG_FPGATEMP", "",
                 0UL, 255UL, 0UL, NULL, 0UL, BX_STAT_DIAG_FPGATEMP ,
            NULL, 0UL, NULL));
        ADDPAR(( "BX_STAT_DIAG_STATUS", "",
            0UL, 255UL, 0UL, NULL, 0UL, BX_STAT_DIAG_STATUS ,
            NULL, 0UL, NULL));
      }

      if (BestXHasFaust(handle))
      {
        /* LM87 */
        ADDPAR(( "BX_STAT_DIAG_BOARDTEMP", "",
            0UL, 255UL, 0UL, NULL, 0UL, BX_STAT_DIAG_BOARDTEMP ,
            NULL, 0UL, NULL));
        
        ADDPAR(( "BX_STAT_DIAG_V5", "",
            0UL, 255UL, 0UL, NULL, 0UL, BX_STAT_DIAG_V5 ,
            NULL, 0UL, NULL));
        
        ADDPAR(( "BX_STAT_DIAG_V12", "",
            0UL, 255UL, 0UL, NULL, 0UL, BX_STAT_DIAG_V12 ,
            NULL, 0UL, NULL));
        
        ADDPAR(( "BX_STAT_DIAG_VIO", "",
            0UL, 255UL, 0UL, NULL, 0UL, BX_STAT_DIAG_VIO ,
            NULL, 0UL, NULL));
        
        ADDPAR(( "BX_STAT_DIAG_VCC", "",
            0UL, 255UL, 0UL, NULL, 0UL, BX_STAT_DIAG_VCC ,
            NULL, 0UL, NULL));
        
        ADDPAR(( "BX_STAT_DIAG_VCORE", "",
            0UL, 255UL, 0UL, NULL, 0UL, BX_STAT_DIAG_VCORE ,
            NULL, 0UL, NULL));
        
        ADDPAR(( "BX_STAT_DIAG_VCCP2", "",
            0UL, 255UL, 0UL, NULL, 0UL, BX_STAT_DIAG_VCCP2 ,
            NULL, 0UL, NULL));
        
        ADDPAR(( "BX_STAT_DIAG_VBOARD", "",
            0UL, 255UL, 0UL, NULL, 0UL, BX_STAT_DIAG_VBOARD ,
            NULL, 0UL, NULL));
        
        ADDPAR(( "BX_STAT_DIAG_FANSPEED", "",
            0UL, 255UL, 0UL, NULL, 0UL, BX_STAT_DIAG_FANSPEED ,
            NULL, 0UL, NULL));
      }
    }
  }
  /* associate the right number of elements */
  pStatusGen->num_elem = num_elem;
  pCapable->info[BX_PARAM_STATUS].paraminfo = p_anchor;  /* set it */

  /* -------------------------------------------------------------------- */
  /* board properties */
  /* -------------------------------------------------------------------- */

  /* short cut */
  pBoardGen = pCapable->info[BX_PARAM_BOARD].geninfo;

  pBoardGen->group_name = "Board";  /* group name */

  p_anchor = NULL;
  num_elem = 0;

  ADDPAR(( "Respect BIOS", "respectbios",
       0UL,                   /* min */
       1UL,                   /* max */
       1UL,                   /* default */
       NULL /* no defStr */,
       BX_PARAMRULE_NORULE,
       BX_BOARD_RESPECTBIOS,
       yesno_table,
       0UL, NULL));

  ADDPAR(( "Bus Clock", "busclock",
       0UL,                     /* min */
       1UL,                     /* max */
       BX_BOARD_BUSCLOCK_DEF,   /* default */
       NULL /* no defStr */,
       BX_PARAMRULE_NORULE,
       BX_BOARD_BUSCLOCK,
       NULL /* no string table */, 0UL, NULL));

  ADDPAR(( "Board Display", "display",
       0UL,                   /* min */
       3UL,                   /* max */
       BX_BOARD_DISPLAY_DEF,  /* default */
       NULL, BX_PARAMRULE_NORULE, BX_BOARD_DISPLAY,
       BX_BOARD_DISPLAY_table, 0UL, NULL));

  ADDPAR(( "Trig IO 0 Mode", "trigio0mode",
       2UL,                     /* min */
       3UL,                     /* max */
       BX_BOARD_TRIGIO_MODE_INPONLY, /* default */
       NULL, BX_PARAMRULE_ZEROALLOWED, BX_BOARD_TRIGIO0_MODE,
       BX_BOARD_TRIGIO_MODE_table, 0UL, NULL));
  
  ADDPAR(( "Trig IO 1 Mode", "trigio1mode",
       2UL,                     /* min */
       3UL,                     /* max */
       BX_BOARD_TRIGIO_MODE_INPONLY, /* default */
       NULL, BX_PARAMRULE_ZEROALLOWED, BX_BOARD_TRIGIO1_MODE,
       BX_BOARD_TRIGIO_MODE_table, 0UL, NULL));
  
  ADDPAR(( "Trig IO 2 Mode", "trigio2mode",
       2UL,                     /* min */
       3UL,                     /* max */
       BX_BOARD_TRIGIO_MODE_INPONLY, /* default */
       NULL, BX_PARAMRULE_ZEROALLOWED, BX_BOARD_TRIGIO2_MODE,
       BX_BOARD_TRIGIO_MODE_table, 0UL, NULL));
  
  ADDPAR(( "Trig IO 3 Mode", "trigio3mode",
       2UL,                     /* min */
       3UL,                     /* max */
       BX_BOARD_TRIGIO_MODE_INPONLY, /* default */
       NULL, BX_PARAMRULE_ZEROALLOWED, BX_BOARD_TRIGIO3_MODE,
       BX_BOARD_TRIGIO_MODE_table, 0UL, NULL));
  
  ADDPAR(( "Trig IO 0 Out", "trigio0out",
       0UL,                     /* min */
       3UL,                     /* max */
       BX_BOARD_TRIGIO_OUT_TRACETRIG, /* default */
       NULL, BX_PARAMRULE_NORULE, BX_BOARD_TRIGIO0_OUT,
       BX_BOARD_TRIGIO_OUT_table, 0UL, NULL));
  
  ADDPAR(( "Trig IO 1 Out", "trigio1out",
       0UL,                     /* min */
       3UL,                     /* max */
       BX_BOARD_TRIGIO_OUT_PROTERR, /* default */
       NULL, BX_PARAMRULE_NORULE, BX_BOARD_TRIGIO1_OUT,
       BX_BOARD_TRIGIO_OUT_table, 0UL, NULL));

  ADDPAR(( "Trig IO 2 Out", "trigio2out",
       0UL,                     /* min */
       3UL,                     /* max */
       BX_BOARD_TRIGIO_OUT_DATACOMP, /* default */
       NULL, BX_PARAMRULE_NORULE, BX_BOARD_TRIGIO2_OUT,
       BX_BOARD_TRIGIO_OUT_table, 0UL, NULL));
  
  ADDPAR(( "Trig IO 3 Out", "trigio3out",
       0UL,                     /* min */
       3UL,                     /* max */
       BX_BOARD_TRIGIO_OUT_TRIGSOURCE, /* default */
       NULL, BX_PARAMRULE_NORULE, BX_BOARD_TRIGIO3_OUT,
       BX_BOARD_TRIGIO_OUT_table, 0UL, NULL));

  if (!BestXHasMephisto(handle))
  {
    ADDPAR(( "PCIXCAP", "pcixcap",
        0UL,                      /* min */
        5UL,                      /* max */
            BX_BOARD_PCIXCAP_PCIX66, /* default */
        NULL, BX_PARAMRULE_NORULE, BX_BOARD_PCIXCAP,
        BX_BOARD_PCIXCAP_table, 0UL, NULL));

    ADDPAR(( "M66EN", "m66en",
        0UL,                     /* min */
        2UL,                     /* max */
            BX_BOARD_M66EN_OPEN,     /* default */
        NULL, BX_PARAMRULE_NORULE, BX_BOARD_M66EN,
        BX_BOARD_M66EN_table, 0UL, NULL));
  }

  pBoardGen->num_elem = num_elem;
  pCapable->info[BX_PARAM_BOARD].paraminfo = p_anchor;

  BX_ERRETURN(BX_E_OK);
}

/* ------------------------------------------------------------------------
 * ------------------------------------------------------------------------ */

/* ------------------------------------------------------------------------
 * The compilers (application) and the GUI  need to know information about
 * the hardware and its capabilities. This information is seperated into
 * a generic info (that is the number of parameters or signals, the depth
 * of the corresponding memory and so on). This information is given here.
 * The calling function gets a generic_infotype struct returned and
 * depending on the typ of information requested the fields within in the
 * struct contain the corresponding information.
 * ------------------------------------------------------------------------ */
static bx_errtype IBestXInfoGet(
    bx_handletype handle,
    bx_paramtype param_typ,      /* what kind of param block at all */
    const bx_infotype ** info,     /* the info struct */
    bx_int8 generate_errors      /* use this to indicate if you want error
                                  * info set */
)
{
  bx_errtype err = BX_E_OK;

  if (! __BX_HANDLECHECK)
  {
    err = BX_E_BAD_HANDLE;
  }
  else if (bx_handlearray[handle].capable == NULL) /* no dyn info -> Error */
  {
    err = BX_E_DYNAMIC_CAPABILITY;
  }
  else if (info == NULL)     /* wrong parameter */
  {
    BX_E_ERROR_MSG_SET("IBestXInfoGet: Parameter 3 (bx_infotype) returned NULL");
    err = BX_E_ERROR;        
  }
  else
  {
    *info = NULL;

    if (param_typ < BX_PARAM_LAST)
    {
      *info = &(bx_handlearray[handle].capable->info[param_typ]);
    }

    if (*info == NULL)         /* not found? */
    {
      err = BX_E_TBD; /* TODO: find better error msg */
    }
  }

  if (generate_errors)
  {
    BX_ERRETURN(err);         /* save error in handle array */
  }

  return err;
}

/*---------------------------------------------------------------------------*
 * EXPORT bx_errtype BestXInfoGet (
 *
 * Purpose    : externally visible version, just calls IBestXInfoGet
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT  BestXInfoGet (
  bx_handletype handle,
  bx_paramtype param,
  const bx_infotype ** info
  )
{
  return IBestXInfoGet(handle, param, info, BX_TRUE);
}

/* ------------------------------------------------------------------------
 * This function deletes the dynamic capabilities and frees all allocated
 * memory.
 * ------------------------------------------------------------------------ */
bx_errtype EXPORT BestXCapaClose(bx_handletype handle)
{
  bx_dynamic_capability_type *pCapable = bx_handlearray[handle].capable;

  if (pCapable != NULL)
  {
    bx_int32 i;            /* general purpos counter */

    for (i=0; i<BX_PARAM_LAST; i++)
    {
      /* delete generic info */
      BestXMemFree( (void**) &(pCapable->info[i].geninfo) );
      if (i != BX_PARAM_TRACE_MEM)  /* tracemem not freed (it is static) */
      {
        /* delete param info */
        BestXMemFree( (void**) &(pCapable->info[i].paraminfo) );
      }
    }
    BestXMemFree ( (void**) &(pCapable->capa_code) );
    BestXMemFree ( (void**) &(pCapable->info) );
    BestXMemFree ( (void**) &(pCapable) );
  }

  BX_ERRETURN(BX_E_OK);
}


/* tries to access the param info structure. In essence checks for the
   existance of a parameter */
bx_errtype EXPORT BestXParamProbe (
    bx_handletype        handle,
    bx_paramtype         param_typ,   /* what kind of parameter at all */
    bx_int32             parnum
    )
{
  return BestXParamInfoGet (handle, param_typ, parnum, NULL, BX_ENUM_SEARCH);
}


/* ------------------------------------------------------------------------
 * This function checks a Parameter if the value is within the correct
 * range. The range depends on hardware and license information.
 * Example: ACK64# can only be set if hardware is Ichiban and license for
 * 64 bit exerciser is valid.
 * In this example only the setting to 1 might be illegal (if exerciser
 * itself is licensed but not the 64 bit option).
 * Illegal is always a setting of a parameter different than the default
 * value if there is no license available.
 * ------------------------------------------------------------------------ */
bx_errtype EXPORT BestXParamCheck(
    bx_handletype handle,
    bx_paramtype param_typ,      /* kind of parameter */
    bx_int32 prop_typ,           /* the parameter itself */
    bx_int32 value)
{                               /* erweiterung um strings noetig !!! */
/*bx_errtype err = BX_E_OK; */
  bx_errtype __status=BX_E_OK;
  bx_int32 index;
  const bx_infotype * myinfo;
  const bx_param_infotype * paramInfo;

  BestXLastErrorAllParamsSet(handle, (bx_int32) param_typ, prop_typ,0UL, 0UL, value);

  /* get pointer to generic info and param info head */
  BX_ERRCHECK(IBestXInfoGet(handle, param_typ, &myinfo,DO_STANDARD_ERROR_HANDLING));

  /* search for specific parameter */
  BX_ERRCHECK(BestXParamSearch (myinfo, prop_typ, BX_ENUM_SEARCH, &index));

  paramInfo = &(myinfo->paraminfo[index]);

  /* check range of specific parameter */
  __status = (BestXParamRangeCheck(handle, paramInfo, value));
  if (__status != BX_E_OK)
  {
    BestXLastErrorParamSet (handle, BX_ERRPAR_1, param_typ);
  }

  BX_ERRETURN (__status);
}

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXParamRangeCheck
 *
 * Purpose: check parameter and value range
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXParamRangeCheck(
  bx_handletype handle,
  const bx_param_infotype * paramInfo,
  bx_int32 value
  )
{
  bx_errtype err = BX_E_OK;

  /* skip special rule checking if it is an observer rule: here, the paramrule
     contains the bit position in the obserr register! */
  if ( paramInfo->paramrule & (bx_int32) BX_PARAMRULE_BITSET )
  {
    /* e.g. decoder speed for Ichiban20 allows values 0,1 or 4 */

    /*
     * RULE:
     *     If the i-th bit (0,..,31) of max_val is set,
     *     the value min_val+i is allowed:
     */

    if (value>paramInfo->min_val+31||value<paramInfo->min_val||
    !(1<<(value-paramInfo->min_val) & paramInfo->max_val))
    {
      BestXLastErrorParamSet(handle, BX_ERRPAR_3, value);
      err = (BX_E_VALUE);
    }
  }
  else if ( paramInfo->paramrule & (bx_int32)BX_PARAMRULE_ZERO2WIDTH )
  {
    /* e.g. Timing Checker (see e2926a_tc_signal_table[]) */
    /* allowed rage is from 0 to 2^width-1; width is stored in max_val */

    bx_int32 MaxVal= ( paramInfo->max_val==32 ?
               0xffffffff :
               (1UL<<paramInfo->max_val)-1 );

    BestXLastErrorAllParamsSet(handle, 0UL, paramInfo->proptyp.value,
                   0, MaxVal, value);

    /* now check if the parameter is within the bounds */
    if (value > MaxVal)
    {
      err = (BX_E_RANGE);
    }
  }
  else                /* -> no special rule handling */
  {
    /* just in case we have an error, set the parameter so the error is
     * traceable and can get restored later */
    BestXLastErrorAllParamsSet(handle, 0UL, paramInfo->proptyp.value,
                   paramInfo->min_val, paramInfo->max_val, value);

    /* check if the parameter is within the bounds */
    if ( ((paramInfo->max_val < value) || (paramInfo->min_val > value)) &&
     !(value == 0 &&
       (paramInfo->paramrule & (bx_int32) BX_PARAMRULE_ZEROALLOWED)) &&
     !(value == 0xffffffffL &&
       (paramInfo->paramrule & (bx_int32) BX_PARAMRULE_MAXLONGALLOWED)) )
    {
      err = (BX_E_RANGE);
    }

    /* now check if the parameter behaves according to the parameter rules */
    /* first check if the parameter is at a dword boundary */
    else if ( (paramInfo->paramrule & (bx_int32)BX_PARAMRULE_DWORDBOUND) &&
          (value & 3UL))
    {
      BestXLastErrorParamSet(handle, BX_ERRPAR_4, value);
      err = (BX_E_ALIGN);
    }
    else if ( (paramInfo->paramrule & (bx_int32)BX_PARAMRULE_MINMAXONLY) &&
          (value != paramInfo->min_val) &&
          (value != paramInfo->max_val))
    {
      BestXLastErrorParamSet(handle, BX_ERRPAR_3, value);
      err = (BX_E_VALUE);
    }
  }

  BX_ERRETURN(err);
}


/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXGenInfoGet(
 *
 * Purpose : the officially exported function, calls internal get with
 *           parameter true for the error generation
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXGenInfoGet(
    bx_handletype handle,
    bx_paramtype param_typ,      /* what kind of param block at all */
    const bx_generic_infotype ** geninfo /* the info struct */
  )
{
  const bx_infotype * info;
  bx_errtype err = IBestXInfoGet(handle, param_typ, &info, BX_TRUE);

  if (err == BX_E_OK)
  {
    if (geninfo)
    {
      *geninfo = info->geninfo;
    }
    else
    {
      err = BX_E_TBD;         /* TODO: find good error msg */
    }
  }

  BX_ERRETURN (err);
}

/* ------------------------------------------------------------------------
 * For the parameters itself this function needs to get called. Using an
 * Id to identify the parameters gives the compilers the ability to access
 * the information.
 * NOTE: if paraminfo is NULL, no info is returned (just param checking)
 * ------------------------------------------------------------------------ */
static bx_errtype IBestXParamInfoGet(
    bx_handletype handle,
    bx_paramtype param_typ,      /* what kind of parameter at all */
    bx_int64 parnum,             /* the parameter itself (as a number) */
    const bx_param_infotype ** paraminfo,  /* the info struct */
    bx_int64 option,             /* how to search for params (index or enum */
    bx_int8 generate_errors      /* do or not do error handling */
)
{
  bx_errtype err = BX_E_OK;
  const bx_infotype *myinfo;

  if (generate_errors == DO_STANDARD_ERROR_HANDLING)
  {
    BX_HANDLECHECK;
    BestXLastErrorAllParamsSet(handle, (bx_int32) param_typ, parnum,
                   0UL, 0UL, 0UL);
  }

  err = IBestXInfoGet (handle, param_typ, &myinfo, BX_FALSE);

  /* no errors yet? */
  if (err == BX_E_OK)
  {
    bx_int32 index;
    /* search for parameter and do some further checking */
    if ( (err = BestXParamSearch (myinfo, parnum, option, &index)) == BX_E_OK )
    {
      if (paraminfo != NULL)
      {
        *paraminfo = &(myinfo->paraminfo[index]);
      }
    }
  }

  if (generate_errors)
  {
    BX_ERRETURN(err);
  }
  else
  {
    return err;
  }
}

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXParamInfoGet(
 *
 * Purpose    :
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXParamInfoGet(
    bx_handletype handle,
    bx_paramtype param_typ,      /* what kind of parameter at all */
    bx_int32 parnum,             /* the parameter itself (as a number) */
    const bx_param_infotype ** paraminfo,  /* the info struct */
    bx_int32 option              /* how to search for params (index or enum */
)
{
  BX_ERRETURN(IBestXParamInfoGet(handle, param_typ, parnum, paraminfo, option,
      DO_STANDARD_ERROR_HANDLING));
}

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXParamSearch
 *
 * Purpose    : search for specific parameter in the param info array
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXParamSearch (const bx_infotype * info,
                    bx_intptr propVal,
                    bx_int64 option,
                    bx_int32 * pIndex)
{
  bx_errtype err = BX_E_OK;
  bx_int32 id;

  if (pIndex == NULL)
  {
    err = BX_E_TBD_MSG;        /* TODO: find good error message */
  }
  else if (option == BX_INDEX_SEARCH) /* just use the index itself */
  {
    id = (bx_int32)propVal;                /* make the index point to propVal */
  }
  else if (option == BX_ENUM_SEARCH)
  {
    /* search the parameter in the list using the enum as identifier */
    for (id = 0; id < info->geninfo->num_elem; id++)
    {
      if (propVal == info->paraminfo[(int) id].proptyp.value)
        break;
    }
  }
  else if (option == BX_NAME_SEARCH)
  {
    /* search the parameter using its name */
    for (id = 0; id < info->geninfo->num_elem; id++)
    {
      if (str_eq(info->paraminfo[(int) id].param_name, (char *) propVal))
        break;
    }
  }
  else if (option == BX_CLINAME_SEARCH)
  {
    /* search the parameter using its CLI name */
    for (id = 0; id < info->geninfo->num_elem; id++)
    {
      if (str_eq(info->paraminfo[(int) id].param_short, (char *) propVal))
        break;
    }
  }
  else
  {
    /* TODO; proper response to this case (invalid option) */
    err = BX_E_TBD_MSG;
    id = 0;
  }

  if (id >= info->geninfo->num_elem)
  {
    *pIndex = 0;
    err = BX_E_PARAM_NOT_EXIST;
  }
  else
  {
    *pIndex = id;
  }

  return err;
}

/* ------------------------------------------------------------------------
 * This function creates error strings out of the data given as parameters.
 * No further information is required. The error structure is not touched.
 * ------------------------------------------------------------------------ */
char *BestXDynamicErrorStringGet(bx_handletype handle,
    bx_int64 group,
    bx_int64 param,
    bx_int64 val1,
    bx_int64 val2,
    bx_int64 val3,
    bx_errtype theerr,
    char *buf)
{
  bx_errtype err;

  const bx_infotype *myinfo;
  const bx_param_infotype *myparaminfo;
  /* get generic info */
  err = IBestXInfoGet(handle,
             (bx_paramtype) group, &myinfo,
             DO_NO_ERROR_HANDLING);
  if (err != BX_E_OK)
  {
    BESTX_SPRINTF(buf,
          "Illegal parameter 'group' (%lu) for "
          "BestXDynamicErrorStringGet"
          ,group);
    return buf;
  }

  /* get param info */
  if (theerr != BX_E_PARAM_NOT_EXIST && theerr != BX_EFW_PARAM_NOT_EXIST)
  {
    err = IBestXParamInfoGet(handle, (bx_paramtype) group, param,
                 &myparaminfo, (bx_int32) BX_ENUM_SEARCH,
                 DO_NO_ERROR_HANDLING);
  }

  if (err != BX_E_OK)
  {
    BESTX_SPRINTF(buf,
          "Illegal parameter 'param' (%lu) for "
          "BestXDynamicErrorStringGet"
          ,param);
    return buf;
  }

  /* note; myparaminfo is guaranteed to be initialized from here on */

  switch (theerr)
  {
  case BX_E_RANGE:
  case BX_EFW_RANGE:
    BESTX_SPRINTF(buf, "Parameter '%s' out of '%s' is out of range. "
      "Valid Range is from 0x%lx to 0x%lx. Value was 0x%lx",
      myparaminfo->param_name, myinfo->geninfo->group_name, val1, val2, val3);
    break;
  case BX_E_ALIGN:
  case BX_EFW_ALIGN:
    BESTX_SPRINTF(buf, "Alignment of Parameter '%s' out of '%s' is invalid. "
      "Value was 0x%lx",
      myparaminfo->param_name, myinfo->geninfo->group_name, val2);
    break;
  case BX_E_VALUE:
  case BX_EFW_VALUE:
    BESTX_SPRINTF(buf, "The Parameter '%s' out of '%s' is invalid. Value was 0x%lx",
      myparaminfo->param_name, myinfo->geninfo->group_name, val1);
    break;
  case BX_E_PARAM_NOT_EXIST:
  case BX_EFW_PARAM_NOT_EXIST:
    BESTX_SPRINTF(buf, "The Parameter '%ld' out of '%s' does not exist.",
      param, myinfo->geninfo->group_name);
    break;
  default:
    BESTX_SPRINTF(buf, "Illegal Errorvalue (%d) for BestXDynamicErrorStringGet",
      (int) theerr);
    break;
  }                             /*lint !e788 */

  return buf;
}
/* ------------------------------------------------------------------------
 * This is a function to look for the valid license. In case the user
 * cannot do the requested task, it returns an error, otherwise it is ok.
 * ------------------------------------------------------------------------ */
bx_errtype EXPORT BestXCapabilityCheck(
    bx_handletype handle,
    bx_int32 capa_code
)
{
  bx_errtype merr = BX_E_NO_CAPABILITY;
  bx_int32 counter;
  BX_HANDLECHECK;

  {
    if (bx_handlearray[handle].capable &&
      bx_handlearray[handle].capable->capa_code &&
      (bx_handlearray[handle].capable->capa_code[0] & capa_code) == capa_code)
    {
      merr = BX_E_OK;
    }
    else
    {
      /* capa_code contains at least one bit, which is not licensed.
         It may also contain licenced bits:
         Consider something like BX_LICENSCHECK(CAPI|OBSERVER).
         In order not to get an errormessage for a licenced bit 
         (which may be at a lower bitposition than the unlicenced bit(s)),
         we remove them:
      */

      if (bx_handlearray[handle].capable && bx_handlearray[handle].capable->capa_code)
      {
        capa_code &= ~(bx_handlearray[handle].capable->capa_code[0]); 
        /* capa_code now contains only the not licensed bit(s) */
      }
      else
      {
        /* We currently have no valid license at all.
           Error message may complain about an available license here ! 
        */
        ;
      }
     
      counter = 0;
      if (capa_code)            /* extract the information what license was
                                 * asked for */
      {
        while ((capa_code & 1UL) == 0)
        {
          counter++;
          capa_code >>= 1;
        }
      }
      BestXLastErrorParamSet(handle, BX_ERRPAR_1, counter);
      merr = BX_E_NO_CAPABILITY;
    }
  }

  BX_ERRETURN(merr);

}
/* ------------------------------------------------------------------------
 * This is a function to read back the complete license information
 * ------------------------------------------------------------------------ */
bx_errtype EXPORT BestXCapabilityRead(
    bx_handletype handle,
    bx_int32 * capa_code
)
{
  BX_HANDLECHECK;

  if (bx_handlearray[handle].capable)
    *capa_code = bx_handlearray[handle].capable->capa_code[0];
  else
    *capa_code = BX_CAPABILITY_NONE;

  BX_ERRETURN(BX_E_OK);
}

/*---------------------------------------------------------------------------*
 * EXPORT void BestXParamValue2String (const bx_param_stringlisttype * thelist,
 *
 * Purpose    : get string value for num value if possible,
 *          print num value otherwise
 *                parameter base states the format of the number,
 *          if no string can be found!
 *---------------------------------------------------------------------------*/
#if !defined(AIX) && !defined(_AIX)
const bx_charptrtype EXPORT BestXParamValue2String (
  const bx_param_stringlisttype * thelist,
  bx_int32 value,
  bx_charptrtype retStr,
  bx_int32 base
  )
#else
bx_charptrtype EXPORT BestXParamValue2String (
  const bx_param_stringlisttype * thelist,
  bx_int32 value,
  bx_charptrtype retStr,
  bx_int32 base
  )
#endif
{
  int i=0;
  assert (retStr);

  *retStr = '\0';

  if (thelist)
  {
    while (thelist[i].name && thelist[i].value != value)
    {
      i++;
    }
  }

  /* any of the following ptrs will be NULL if not found... */
  if (thelist && thelist[i].name && *(thelist[i].name))
  {
    BESTX_STRCPY (retStr, thelist[i].name);
  }
  else
  {
    PrintBase (retStr, value, base, 0 /* width */);
    if (base == 16)
    {
      strcat (retStr, "\\h");
    }
    else if (base == 2)
    {
      strcat (retStr, "\\b");
    }
  }

  return retStr;
}

/*---------------------------------------------------------------------------*
 * EXPORT void BestXParamString2Value
 *
 * Purpose    : convert string into value
 * NOTE: changed this to work w/o string list to just convert value!      
 *---------------------------------------------------------------------------*/
#if !defined(AIX) && !defined(_AIX)
const bx_charptrtype EXPORT BestXParamString2Value (
  const bx_param_stringlisttype * thelist,
  bx_ccharptrtype valStr,   /* IN  */
  bx_int32 *pValue        /* OUT */
)
#else
bx_charptrtype EXPORT BestXParamString2Value (
  const bx_param_stringlisttype * thelist,
  bx_ccharptrtype valStr,   /* IN  */
  bx_int32 *pValue        /* OUT */
)
#endif
{
  int i=0;
  assert (valStr);
  assert(pValue);

  *pValue=0;

  while ( thelist && thelist[i].name && strcasecmp (thelist[i].name, valStr) )
  {
    i++;
  }

  /* any of the following ptrs will be NULL if not found... */
  if ( thelist && thelist[i].name )
  {
    *pValue=thelist[i].value;
  }
  else
  {
    bx_charptrtype errPtr = NULL;
    /*  Not found in stringlist, string is something like "34" */
    *pValue=Str2ULong(valStr,&errPtr,0);
    assert (errPtr == NULL || *errPtr == '\0');
  }

  return NULL;
}

/*---------------------------------------------------------------------------*
 * void EXPORT BestXParamValueListGet ( const bx_param_infotype * pParamInfo,
 *
 * Purpose: 
 *---------------------------------------------------------------------------*/
void EXPORT BestXParamValueListGet (
  const bx_param_infotype * pParamInfo,
  bx_int32 minVal, /*  default: 0 */
  bx_int32 maxVal, /*  default: 0xFFFFFFFF */
  bx_ccharptrtype separator,
  bx_charptrtype buffer
  )
{
  bx_int32 i;            /* general purpose counter */
  bx_int32 val;
  char numBuffer[256];
  bx_int32 therule = pParamInfo->paramrule;
  bx_int32 localMin, localMax;

  *buffer = '\0';        /* empty buffer */

  if (therule & BX_PARAMRULE_BITSET) /* special handling for a bitset */
  {
    /*
     * BITSET range rule:
     *     If the i-th bit (0,..,31) of max_val is set,
     *     the value min_val+i is allowed:
     */
    /* NOTE: (HL, 16-Aug-00)
       minVal and maxVal parameters here are NOT using any bitset coding
       but are interpreted as absolute values! */

    for (i = 0; i < 32UL; i++)
    {
      val = i + pParamInfo->min_val;
      
      if ( ( (1UL<<i) & pParamInfo->max_val ) &&
       (val >= minVal) &&
       (val <= maxVal) )
      {
        if (*buffer)        /* already something in there? */
        {
          BESTX_STRCAT(buffer, separator);    /* add separator, e.g. comma */
        }
        BESTX_STRCAT ( buffer,BestXParamValue2String (pParamInfo->stringlist,val, numBuffer, 10) );
      }
    }
  }
  else if (therule & BX_PARAMRULE_ZERO2WIDTH)
  {
    /* CAUTION(chris) : This part is not yet tested !!!!
       (but also currently not used) */
    
    /* Allowed rage is from 0 to 2^width-1; width is stored in maxVal */
    
    localMax = pParamInfo->max_val == 32UL ?
               0xffffffffUL : (1UL << pParamInfo->max_val) - 1UL;
    localMax = min (maxVal, localMax);

    for (val = 0; val <= localMax; val++)
    {
      if (*buffer)        /* already something in there? */
      {
        BESTX_STRCAT(buffer, separator);    /* add separator, e.g. comma */
      }
      BESTX_STRCAT ( buffer,BestXParamValue2String (pParamInfo->stringlist,val, numBuffer, 10) );
    }
  }
  else                /* the other rules basically are lists... */
  {
    bx_int32 increment;
    
    /* do we need to add the value '0'? */
    if (therule & BX_PARAMRULE_ZEROALLOWED)
    {
      /*  Additionally to the range minVal to maxVal, the value */
      /*  zero is allowed.  */
      
      if (*buffer)        /* already something in there? */
      {
        BESTX_STRCAT(buffer, separator);    /* add separator, e.g. comma */
      }
 
      BESTX_STRCAT (buffer,
            BestXParamValue2String (pParamInfo->stringlist,
                        0UL, numBuffer, 10));
    }

    /* any specialities concerning bounds? */
    if (therule & BX_PARAMRULE_128BOUND)
    {
      increment = 128UL;
    }
    if (therule & BX_PARAMRULE_DWORDBOUND)
    {
      increment = 4UL;
    }
    if (therule & BX_PARAMRULE_EVEN)
    {
      increment = 2UL;
    }
    else
    {
      increment = 1UL;
    }
    
    /* adjust min and max for specialities concerning boundaries */
    localMin = max (minVal, pParamInfo->min_val);
    localMax = min (maxVal, pParamInfo->max_val);

    if (localMin % increment != 0UL)
    {
      localMin = (localMin + increment) - (localMin % increment);
    }

    if (localMax % increment != 0UL)
    {
      localMax -= localMax % increment;
    }

    /* go fill the list */
    for (val=localMin; val<=localMax; val++)
    {
      if (*buffer)        /* already something in there? */
      {
        BESTX_STRCAT(buffer, separator);    /* add separator, e.g. comma */
      }
      BESTX_STRCAT (buffer,
            BestXParamValue2String (pParamInfo->stringlist,
                        val, numBuffer, 10));
    }

    /* do we need to add the value of MAXLONG? */
    if (therule & BX_PARAMRULE_MAXLONGALLOWED)
    {
      if (maxVal == 0xFFFFFFFFUL)
      {
        if (*buffer)        /* already something in there? */
        {
          BESTX_STRCAT(buffer, separator);    /* add separator, e.g. comma */
        }
        BESTX_STRCAT (buffer,
              BestXParamValue2String (pParamInfo->stringlist,
                          0xFFFFFFFF, numBuffer, 10));
      }
    }
  }
}

#define SAFE_STR(a) ((a) ? (a) : "")

/*---------------------------------------------------------------------------*
 * bx_errtype BestXPrintDefaults
 *
 * Purpose    : print default values for all properties in dyncap
 *---------------------------------------------------------------------------*/
bx_errtype BestXPrintDefaults (bx_handletype handle, bx_charptrtype fileName)
{
  BX_TRY_VARS;
  FILE * fp = NULL;

  BX_TRY_BEGIN {
    char strBuf[50];
    bx_charptrtype defStr;
    bx_paramtype param;

    if ( (fp = BESTX_FOPEN(fileName, "wb")) == NULL )
    {
      BX_TRY_FAIL (BX_E_FILE_OPEN);
    }
    else
    {
      BX_TRY_PROGRESS (BX_E_OK);
    }

    BESTX_FPRINTF (fp, "Default values for all CAPI properties: \n\n");

    BESTX_FPRINTF (fp, "  %-25s (CAPI) <CLI> : %-20s (%s)\n",
           "parameter name",
           "default name",
           "corrsponding default value" );

    for (param=0; param<BX_PARAM_LAST; param++)
    {
      bx_int32 index = 0;
      bx_errtype err = BX_E_OK;
      const bx_generic_infotype * geninfo;
      const bx_param_infotype * paraminfo;
      
      BX_TRY (BestXGenInfoGet(handle, param, &geninfo));

      BESTX_FPRINTF (fp, "\nGroup %s:\n", geninfo->group_name);

      while (index < geninfo->num_elem)
      {
        err = BestXParamInfoGet(handle, param, index,
          &paraminfo, BX_INDEX_SEARCH);

        if (err == BX_E_OK)
        {
          if (paraminfo->defaultstr)
          {
            defStr = paraminfo->defaultstr;
          }
          else
          {
            defStr = BestXParamValue2String(paraminfo->stringlist,
              paraminfo->defaultval,
              strBuf, 10);
          }

          BESTX_FPRINTF (fp, "  %-25s (%s) <%s> : %s (%ld)\n",
            SAFE_STR(paraminfo->param_name),
            SAFE_STR(paraminfo->param_propname),
            SAFE_STR(paraminfo->param_short),
            SAFE_STR(defStr),
            paraminfo->defaultval);
          index++;
        }
      }
    }

    BX_TRY_FAIL (BX_E_INTERNAL_RETURN);
  }

  BX_TRY_CATCH {
    BX_TRY_PASSED {        /* was file opened successfully? */
      fclose (fp);
    }

    if (BX_TRY_RET == BX_E_INTERNAL_RETURN)
    {
      BX_TRY_RET = BX_E_OK;
    }
  }

  BX_ERRETURN (BX_TRY_RET);
}
